"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleMigrationsTaskClient = void 0;
var _constants = require("../../../../../common/siem_migrations/constants");
var _rule_migrations_task_runner = require("./rule_migrations_task_runner");
var _rule_migrations_task_evaluator = require("./rule_migrations_task_evaluator");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleMigrationsTaskClient {
  constructor(migrationsRunning, logger, data, currentUser, dependencies) {
    this.migrationsRunning = migrationsRunning;
    this.logger = logger;
    this.data = data;
    this.currentUser = currentUser;
    this.dependencies = dependencies;
  }

  /** Starts a rule migration task */
  async start(params) {
    var _invocationConfig$con;
    const {
      migrationId,
      connectorId,
      invocationConfig
    } = params;
    if (this.migrationsRunning.has(migrationId)) {
      return {
        exists: true,
        started: false
      };
    }
    // Just in case some previous execution was interrupted without cleaning up
    await this.data.rules.updateStatus(migrationId, {
      status: _constants.SiemMigrationStatus.PROCESSING
    }, _constants.SiemMigrationStatus.PENDING, {
      refresh: true
    });
    const {
      rules
    } = await this.data.rules.getStats(migrationId);
    if (rules.total === 0) {
      return {
        exists: false,
        started: false
      };
    }
    if (rules.pending === 0) {
      return {
        exists: true,
        started: false
      };
    }
    const migrationLogger = this.logger.get(migrationId);
    const abortController = new AbortController();
    const migrationTaskRunner = new _rule_migrations_task_runner.RuleMigrationTaskRunner(migrationId, this.currentUser, abortController, this.data, migrationLogger, this.dependencies);
    await migrationTaskRunner.setup(connectorId);
    if (this.migrationsRunning.has(migrationId)) {
      // Just to prevent a race condition in the setup
      throw new Error('Task already running for this migration');
    }
    migrationLogger.info('Starting migration');
    this.migrationsRunning.set(migrationId, migrationTaskRunner);
    await this.data.migrations.saveAsStarted({
      id: migrationId,
      connectorId,
      skipPrebuiltRulesMatching: (_invocationConfig$con = invocationConfig.configurable) === null || _invocationConfig$con === void 0 ? void 0 : _invocationConfig$con.skipPrebuiltRulesMatching
    });

    // run the migration in the background without awaiting and resolve the `start` promise
    migrationTaskRunner.run(invocationConfig).then(() => {
      // The task runner has finished normally. Abort errors are also handled here, it's an expected finish scenario, nothing special should be done.
      migrationLogger.debug('Migration execution task finished');
      this.data.migrations.saveAsFinished({
        id: migrationId
      }).catch(error => {
        migrationLogger.error(`Error saving migration as finished: ${error}`);
      });
    }).catch(error => {
      // Unexpected errors, no use in throwing them since the `start` promise is long gone. Just log and store the error message
      migrationLogger.error(`Error executing migration task: ${error}`);
      this.data.migrations.saveAsFailed({
        id: migrationId,
        error: error.message
      }).catch(saveError => {
        migrationLogger.error(`Error saving migration as failed: ${saveError}`);
      });
    }).finally(() => {
      this.migrationsRunning.delete(migrationId);
    });
    return {
      exists: true,
      started: true
    };
  }

  /** Updates all the rules in a migration to be re-executed */
  async updateToRetry(migrationId, filter) {
    if (this.migrationsRunning.has(migrationId)) {
      // not update migrations that are currently running
      return {
        updated: false
      };
    }
    filter.installed = false; // only retry rules that are not installed
    await this.data.rules.updateStatus(migrationId, filter, _constants.SiemMigrationStatus.PENDING, {
      refresh: true
    });
    return {
      updated: true
    };
  }

  /** Returns the stats of a migration */
  async getStats(migrationId) {
    const migration = await this.data.migrations.get({
      id: migrationId
    });
    if (!migration) {
      throw new Error(`Migration with ID ${migrationId} not found`);
    }
    const dataStats = await this.data.rules.getStats(migrationId);
    const taskStats = this.getTaskStats(migration, dataStats.rules);
    return {
      ...taskStats,
      ...dataStats,
      name: migration.name
    };
  }

  /** Returns the stats of all migrations */
  async getAllStats() {
    const allDataStats = await this.data.rules.getAllStats();
    const allMigrations = await this.data.migrations.getAll();
    const allMigrationsMap = new Map(allMigrations.map(migration => [migration.id, migration]));
    const allStats = [];
    for (const dataStats of allDataStats) {
      const migration = allMigrationsMap.get(dataStats.id);
      if (migration) {
        const tasksStats = this.getTaskStats(migration, dataStats.rules);
        allStats.push({
          ...tasksStats,
          ...dataStats,
          name: migration.name
        });
      }
    }
    return allStats;
  }
  getTaskStats(migration, dataStats) {
    return {
      status: this.getTaskStatus(migration, dataStats),
      last_execution: migration.last_execution
    };
  }
  getTaskStatus(migration, dataStats) {
    const {
      id: migrationId,
      last_execution: lastExecution
    } = migration;
    if (this.migrationsRunning.has(migrationId)) {
      return _constants.SiemMigrationTaskStatus.RUNNING;
    }
    if (dataStats.completed + dataStats.failed === dataStats.total) {
      return _constants.SiemMigrationTaskStatus.FINISHED;
    }
    if (lastExecution !== null && lastExecution !== void 0 && lastExecution.is_stopped) {
      return _constants.SiemMigrationTaskStatus.STOPPED;
    }
    if (dataStats.pending === dataStats.total) {
      return _constants.SiemMigrationTaskStatus.READY;
    }
    return _constants.SiemMigrationTaskStatus.INTERRUPTED;
  }

  /** Stops one running migration */
  async stop(migrationId) {
    try {
      const migrationRunning = this.migrationsRunning.get(migrationId);
      if (migrationRunning) {
        migrationRunning.abortController.abort('Stopped by user');
        await this.data.migrations.setIsStopped({
          id: migrationId
        });
        return {
          exists: true,
          stopped: true
        };
      }
      const {
        rules
      } = await this.data.rules.getStats(migrationId);
      if (rules.total > 0) {
        return {
          exists: true,
          stopped: true
        };
      }
      return {
        exists: false,
        stopped: true
      };
    } catch (err) {
      this.logger.error(`Error stopping migration ID:${migrationId}`, err);
      return {
        exists: true,
        stopped: false
      };
    }
  }

  /** Creates a new evaluator for the rule migration task */
  async evaluate(params) {
    const {
      evaluationId,
      langsmithOptions,
      connectorId,
      invocationConfig,
      abortController
    } = params;
    const migrationLogger = this.logger.get('evaluate');
    const migrationTaskEvaluator = new _rule_migrations_task_evaluator.RuleMigrationTaskEvaluator(evaluationId, this.currentUser, abortController, this.data, migrationLogger, this.dependencies);
    await migrationTaskEvaluator.evaluate({
      connectorId,
      langsmithOptions,
      invocationConfig
    });
  }

  /** Returns if a migration is running or not */
  isMigrationRunning(migrationId) {
    return this.migrationsRunning.has(migrationId);
  }
}
exports.RuleMigrationsTaskClient = RuleMigrationsTaskClient;