"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleResourceRetriever = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _resources = require("../../../../../../common/siem_migrations/rules/resources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleResourceRetriever {
  constructor(migrationId, dataClient) {
    (0, _defineProperty2.default)(this, "existingResources", void 0);
    this.migrationId = migrationId;
    this.dataClient = dataClient;
  }
  async initialize() {
    const batches = this.dataClient.resources.searchBatches(this.migrationId, {
      filters: {
        hasContent: true
      }
    } // filters out missing (undefined) content resources, empty strings content will be included
    );
    const existingRuleResources = {
      macro: {},
      lookup: {}
    };
    let resources;
    do {
      resources = await batches.next();
      resources.forEach(resource => {
        existingRuleResources[resource.type][resource.name] = resource;
      });
    } while (resources.length > 0);
    this.existingResources = existingRuleResources;
  }
  async getResources(migrationRule) {
    const originalRule = migrationRule.original_rule;
    const existingResources = this.existingResources;
    if (!existingResources) {
      throw new Error('initialize must be called before calling getResources');
    }
    const resourceIdentifier = new _resources.ResourceIdentifier(originalRule.vendor);
    const resourcesIdentifiedFromRule = resourceIdentifier.fromOriginalRule(originalRule);
    const macrosFound = new Map();
    const lookupsFound = new Map();
    resourcesIdentifiedFromRule.forEach(resource => {
      const existingResource = existingResources[resource.type][resource.name];
      if (existingResource) {
        if (resource.type === 'macro') {
          macrosFound.set(resource.name, existingResource);
        } else if (resource.type === 'lookup') {
          lookupsFound.set(resource.name, existingResource);
        }
      }
    });
    const resourcesFound = [...macrosFound.values(), ...lookupsFound.values()];
    if (!resourcesFound.length) {
      return {};
    }
    let nestedResourcesFound = resourcesFound;
    do {
      const nestedResourcesIdentified = resourceIdentifier.fromResources(nestedResourcesFound);
      nestedResourcesFound = [];
      nestedResourcesIdentified.forEach(resource => {
        const existingResource = existingResources[resource.type][resource.name];
        if (existingResource) {
          nestedResourcesFound.push(existingResource);
          if (resource.type === 'macro') {
            macrosFound.set(resource.name, existingResource);
          } else if (resource.type === 'lookup') {
            lookupsFound.set(resource.name, existingResource);
          }
        }
      });
    } while (nestedResourcesFound.length > 0);
    return {
      ...(macrosFound.size > 0 ? {
        macro: this.formatOutput(macrosFound)
      } : {}),
      ...(lookupsFound.size > 0 ? {
        lookup: this.formatOutput(lookupsFound)
      } : {})
    };
  }
  formatOutput(resources) {
    return Array.from(resources.values()).map(({
      name,
      content,
      type
    }) => ({
      name,
      content,
      type
    }));
  }
}
exports.RuleResourceRetriever = RuleResourceRetriever;