"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EngineDescriptorClient = void 0;
var _fp = require("lodash/fp");
var _engine_descriptor_type = require("./engine_descriptor_type");
var _utils = require("../utils");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EngineDescriptorClient {
  constructor(deps) {
    this.deps = deps;
  }
  getSavedObjectId(entityType) {
    return `entity-engine-descriptor-${entityType}-${this.deps.namespace}`;
  }
  async init(entityType, options) {
    const opts = (0, _fp.merge)(_constants.defaultOptions, options);
    const engineDescriptor = await this.find(entityType);
    if (engineDescriptor.total > 1) {
      throw new Error(`Found multiple engine descriptors for entity type ${entityType}`);
    }
    if (engineDescriptor.total === 1) {
      const old = engineDescriptor.saved_objects[0].attributes;
      const update = {
        ...old,
        error: undefined,
        // if the engine is being re-initialized, clear any previous error
        status: _constants.ENGINE_STATUS.INSTALLING,
        ...opts
      };
      await this.deps.soClient.update(_engine_descriptor_type.entityEngineDescriptorTypeName, this.getSavedObjectId(entityType), update, {
        refresh: 'wait_for'
      });
      return update;
    }
    const {
      attributes
    } = await this.deps.soClient.create(_engine_descriptor_type.entityEngineDescriptorTypeName, {
      status: _constants.ENGINE_STATUS.INSTALLING,
      type: entityType,
      ...opts
    }, {
      id: this.getSavedObjectId(entityType)
    });
    return attributes;
  }
  async update(entityType, engine) {
    const id = this.getSavedObjectId(entityType);
    const {
      attributes
    } = await this.deps.soClient.update(_engine_descriptor_type.entityEngineDescriptorTypeName, id, engine, {
      refresh: 'wait_for'
    });
    return attributes;
  }
  async updateStatus(entityType, status) {
    return this.update(entityType, {
      status
    });
  }
  async find(entityType) {
    return this.deps.soClient.find({
      type: _engine_descriptor_type.entityEngineDescriptorTypeName,
      filter: (0, _utils.getByEntityTypeQuery)(entityType),
      namespaces: [this.deps.namespace]
    });
  }
  async get(entityType) {
    const id = this.getSavedObjectId(entityType);
    const {
      attributes
    } = await this.deps.soClient.get(_engine_descriptor_type.entityEngineDescriptorTypeName, id);
    return attributes;
  }
  async maybeGet(entityType) {
    try {
      const descriptor = await this.get(entityType);
      return descriptor;
    } catch (e) {
      if (e.isBoom && e.output.statusCode === 404) {
        return undefined;
      }
      throw e;
    }
  }
  async list() {
    return this.deps.soClient.find({
      type: _engine_descriptor_type.entityEngineDescriptorTypeName,
      namespaces: [this.deps.namespace]
    }).then(({
      saved_objects: engines
    }) => ({
      engines: engines.map(engine => engine.attributes),
      count: engines.length
    }));
  }
  async delete(entityType) {
    const id = this.getSavedObjectId(entityType);
    return this.deps.soClient.delete(_engine_descriptor_type.entityEngineDescriptorTypeName, id);
  }
}
exports.EngineDescriptorClient = EngineDescriptorClient;