"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetCriticalityMigrationClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _asset_criticality_data_client = require("./asset_criticality_data_client");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ECS_MAPPINGS_MIGRATION_QUERY = {
  bool: {
    must_not: {
      exists: {
        field: 'asset.criticality'
      }
    }
  }
};
const PAINLESS_SCRIPT = `
Map asset = new HashMap();
asset.put('criticality', ctx._source.criticality_level);
ctx._source.asset = asset;
if (ctx._source.id_field == 'user.name') {
  Map user = new HashMap();
  user.put('name', ctx._source.id_value);
  user.put('asset', asset);
  ctx._source.user = user;
} else {
  Map host = new HashMap();
  host.put('name', ctx._source.id_value);
  host.put('asset', asset);
  ctx._source.host = host;
}`;
class AssetCriticalityMigrationClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "assetCriticalityDataClient", void 0);
    (0, _defineProperty2.default)(this, "isMappingsMigrationRequired", async () => {
      const indicesMappings = await this.assetCriticalityDataClient.getIndexMappings();
      return Object.values(indicesMappings).some(({
        mappings
      }) => {
        var _mappings$_meta;
        return ((_mappings$_meta = mappings._meta) === null || _mappings$_meta === void 0 ? void 0 : _mappings$_meta.version) !== _constants.ASSET_CRITICALITY_MAPPINGS_VERSIONS;
      });
    });
    (0, _defineProperty2.default)(this, "isEcsDataMigrationRequired", async () => {
      const resp = await this.assetCriticalityDataClient.search({
        query: ECS_MAPPINGS_MIGRATION_QUERY,
        size: 1
      });
      return resp.hits.hits.length > 0;
    });
    (0, _defineProperty2.default)(this, "migrateMappings", async () => {
      const spaceIds = await this.getAllSpacesWithAssetCriticalityInstalled();
      for (const spaceId of spaceIds) {
        const assetCriticalityDataClient = new _asset_criticality_data_client.AssetCriticalityDataClient({
          ...this.options,
          namespace: spaceId
        });
        try {
          await assetCriticalityDataClient.createOrUpdateIndex();
        } catch (error) {
          this.options.logger.error(`Failed to create or update index for space ${spaceId}: ${error.message}`);
        }
      }
    });
    (0, _defineProperty2.default)(this, "migrateEcsData", abortSignal => {
      return this.options.esClient.updateByQuery({
        index: this.assetCriticalityDataClient.getIndex(),
        conflicts: 'proceed',
        ignore_unavailable: true,
        allow_no_indices: true,
        scroll_size: 10000,
        body: {
          query: ECS_MAPPINGS_MIGRATION_QUERY,
          script: {
            source: PAINLESS_SCRIPT,
            lang: 'painless'
          }
        }
      }, {
        requestTimeout: '5m',
        retryOnTimeout: true,
        maxRetries: 2,
        signal: abortSignal
      });
    });
    (0, _defineProperty2.default)(this, "copyTimestampToEventIngestedForAssetCriticality", abortSignal => {
      return this.options.esClient.updateByQuery({
        index: this.assetCriticalityDataClient.getIndex(),
        conflicts: 'proceed',
        ignore_unavailable: true,
        allow_no_indices: true,
        body: {
          query: {
            bool: {
              must_not: {
                exists: {
                  field: 'event.ingested'
                }
              }
            }
          },
          script: {
            source: 'ctx._source.event.ingested = ctx._source.@timestamp',
            lang: 'painless'
          }
        }
      }, {
        requestTimeout: '5m',
        retryOnTimeout: true,
        maxRetries: 2,
        signal: abortSignal
      });
    });
    (0, _defineProperty2.default)(this, "getAllSpacesWithAssetCriticalityInstalled", async () => {
      const indicesMappings = await this.assetCriticalityDataClient.getIndexMappings();
      const indices = Object.keys(indicesMappings);

      // we remove the last character from the index name to get the base index name
      // the last character is * to match all spaces
      const indexBase = this.assetCriticalityDataClient.getIndex().slice(0, -1);
      const namespaces = [];
      for (const index of indices) {
        const maybeNamespace = index.split(indexBase).at(-1);
        if (maybeNamespace) {
          namespaces.push(maybeNamespace);
        } else {
          this.options.logger.warn(`Index ${index} does not follow the expected naming convention for asset criticality indices.`);
        }
      }
      return namespaces;
    });
    this.options = options;
    this.assetCriticalityDataClient = new _asset_criticality_data_client.AssetCriticalityDataClient({
      ...options,
      namespace: '*' // The migration is applied to all spaces
    });
  }
}
exports.AssetCriticalityMigrationClient = AssetCriticalityMigrationClient;