"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetCriticalityDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/alerting-plugin/common");
var _esQuery = require("@kbn/es-query");
var _create_or_update_index = require("../utils/create_or_update_index");
var _asset_criticality = require("../../../../common/entity_analytics/asset_criticality");
var _constants = require("./constants");
var _audit = require("./audit");
var _audit2 = require("../audit");
var _helpers = require("./helpers");
var _event_ingested_pipeline = require("../utils/event_ingested_pipeline");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_CRITICALITY_RESPONSE_SIZE = 100_000;
const DEFAULT_CRITICALITY_RESPONSE_SIZE = 1_000;
const createId = ({
  idField,
  idValue
}) => `${idField}:${idValue}`;
class AssetCriticalityDataClient {
  constructor(options) {
    /**
     * Bulk upsert asset criticality records from a stream.
     * @param recordsStream a stream of records to upsert, records may also be an error e.g if there was an error parsing
     * @param flushBytes how big elasticsearch bulk requests should be before they are sent
     * @param retries the number of times to retry a failed bulk request
     * @param streamIndexStart By default the errors are zero-indexed. You can change it by setting this param to a value like `1`. It could be useful for file upload.
     * @returns an object containing the number of records updated, created, errored, and the total number of records processed
     * @throws an error if the stream emits an error
     * @remarks
     * - The stream must emit records in the format of {@link AssetCriticalityUpsert} or an error instance
     * - The stream must emit records in the order they should be upserted
     * - The stream must emit records in a valid JSON format
     * - We allow errors to be emitted in the stream to allow for partial upserts and to maintain the order of records
     **/
    (0, _defineProperty2.default)(this, "bulkUpsertFromStream", async ({
      recordsStream,
      flushBytes,
      retries,
      streamIndexStart = 0
    }) => {
      const errors = [];
      const stats = {
        successful: 0,
        failed: 0,
        total: 0
      };
      let streamIndex = streamIndexStart;
      const recordGenerator = async function* () {
        const processedEntities = new Set();
        for await (const untypedRecord of recordsStream) {
          const record = untypedRecord;
          stats.total++;
          if (record instanceof Error) {
            stats.failed++;
            errors.push({
              message: record.message,
              index: streamIndex
            });
          } else {
            const entityKey = `${record.idField}-${record.idValue}`;
            if (processedEntities.has(entityKey)) {
              errors.push({
                message: 'Duplicated entity',
                index: streamIndex
              });
              stats.failed++;
            } else {
              processedEntities.add(entityKey);
              yield {
                record,
                index: streamIndex
              };
            }
          }
          streamIndex++;
        }
      };
      const {
        failed,
        successful
      } = await this.options.esClient.helpers.bulk({
        datasource: recordGenerator(),
        index: this.getIndex(),
        flushBytes,
        retries,
        refreshOnCompletion: this.getIndex(),
        onDocument: ({
          record
        }) => [{
          update: {
            _id: createId(record)
          }
        }, {
          doc: {
            id_field: record.idField,
            id_value: record.idValue,
            criticality_level: record.criticalityLevel,
            asset: {
              criticality: record.criticalityLevel
            },
            ...(0, _helpers.getImplicitEntityFields)(record),
            '@timestamp': new Date().toISOString()
          },
          doc_as_upsert: true
        }],
        onDrop: ({
          document,
          error
        }) => {
          errors.push({
            message: (error === null || error === void 0 ? void 0 : error.reason) || 'Unknown error',
            index: document.index
          });
        }
      });
      stats.successful += successful;
      stats.failed += failed;
      return {
        errors,
        stats
      };
    });
    this.options = options;
  }

  /**
   * Initialize asset criticality resources.
   */
  async init() {
    var _this$options$auditLo;
    await (0, _event_ingested_pipeline.createEventIngestedPipeline)(this.options.esClient, this.options.namespace);
    await this.createOrUpdateIndex();
    (_this$options$auditLo = this.options.auditLogger) === null || _this$options$auditLo === void 0 ? void 0 : _this$options$auditLo.log({
      message: 'User installed asset criticality Elasticsearch resources',
      event: {
        action: _audit.AssetCriticalityAuditActions.ASSET_CRITICALITY_INITIALIZE,
        category: _audit2.AUDIT_CATEGORY.DATABASE,
        type: _audit2.AUDIT_TYPE.CREATION,
        outcome: _audit2.AUDIT_OUTCOME.SUCCESS
      }
    });
  }

  /**
   * It will create idex for asset criticality or update mappings if index exists
   */
  async createOrUpdateIndex() {
    await (0, _create_or_update_index.createOrUpdateIndex)({
      esClient: this.options.esClient,
      logger: this.options.logger,
      options: {
        index: this.getIndex(),
        mappings: {
          ...(0, _common.mappingFromFieldMap)(_constants.assetCriticalityFieldMap, 'strict'),
          _meta: {
            version: _constants.ASSET_CRITICALITY_MAPPINGS_VERSIONS
          }
        },
        settings: {
          default_pipeline: (0, _event_ingested_pipeline.getIngestPipelineName)(this.options.namespace)
        }
      }
    });
  }

  /**
   *
   * A general method for searching asset criticality records.
   * @param query an ESL query to filter criticality results
   * @param size the maximum number of records to return. Cannot exceed {@link MAX_CRITICALITY_RESPONSE_SIZE}. If unspecified, will default to {@link DEFAULT_CRITICALITY_RESPONSE_SIZE}.
   * @returns criticality records matching the query
   */
  async search({
    query,
    size = DEFAULT_CRITICALITY_RESPONSE_SIZE,
    from,
    sort = ['@timestamp'] // without a default sort order the results are not deterministic which makes testing hard
  }) {
    const response = await this.options.esClient.search({
      index: this.getIndex(),
      ignore_unavailable: true,
      query,
      size: Math.min(size, MAX_CRITICALITY_RESPONSE_SIZE),
      from,
      sort,
      post_filter: {
        bool: {
          must_not: {
            term: {
              criticality_level: _constants.CRITICALITY_VALUES.DELETED
            }
          }
        }
      }
    });
    return response;
  }
  async searchByKuery({
    kuery,
    size,
    from,
    sort
  }) {
    const query = kuery ? (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(kuery)) : {
      match_all: {}
    };
    return this.search({
      query,
      size,
      from,
      sort
    });
  }
  getIndex() {
    return (0, _asset_criticality.getAssetCriticalityIndex)(this.options.namespace);
  }
  async doesIndexExist() {
    try {
      var _this$options$auditLo2;
      const result = await this.options.esClient.indices.exists({
        index: this.getIndex()
      });
      (_this$options$auditLo2 = this.options.auditLogger) === null || _this$options$auditLo2 === void 0 ? void 0 : _this$options$auditLo2.log({
        message: 'User checked if the asset criticality Elasticsearch resources were installed',
        event: {
          action: _audit.AssetCriticalityAuditActions.ASSET_CRITICALITY_INITIALIZE,
          category: _audit2.AUDIT_CATEGORY.DATABASE,
          type: _audit2.AUDIT_TYPE.ACCESS,
          outcome: _audit2.AUDIT_OUTCOME.SUCCESS
        }
      });
      return result;
    } catch (e) {
      return false;
    }
  }
  async getStatus() {
    const isAssetCriticalityResourcesInstalled = await this.doesIndexExist();
    return {
      isAssetCriticalityResourcesInstalled
    };
  }
  getIndexMappings() {
    return this.options.esClient.indices.getMapping({
      index: this.getIndex()
    }, {
      ignore: [404]
    });
  }
  async get(idParts) {
    const id = createId(idParts);
    try {
      const {
        _source: doc
      } = await this.options.esClient.get({
        id,
        index: this.getIndex()
      });
      if ((doc === null || doc === void 0 ? void 0 : doc.criticality_level) === _constants.CRITICALITY_VALUES.DELETED) {
        return undefined;
      }
      return doc;
    } catch (err) {
      if (err.statusCode === 404) {
        return undefined;
      } else {
        throw err;
      }
    }
  }
  async upsert(record, refresh = 'wait_for') {
    const id = createId(record);
    const doc = {
      id_field: record.idField,
      id_value: record.idValue,
      criticality_level: record.criticalityLevel,
      '@timestamp': new Date().toISOString(),
      asset: {
        criticality: record.criticalityLevel
      },
      ...(0, _helpers.getImplicitEntityFields)(record)
    };
    await this.options.esClient.update({
      id,
      index: this.getIndex(),
      refresh: refresh !== null && refresh !== void 0 ? refresh : false,
      body: {
        doc,
        doc_as_upsert: true
      }
    });
    return doc;
  }
  async delete(idParts, refresh = 'wait_for') {
    let record;
    try {
      record = await this.get(idParts);
    } catch (err) {
      if (err.statusCode === 404) {
        return undefined;
      } else {
        throw err;
      }
    }
    if (!record) {
      return undefined;
    }
    try {
      await this.options.esClient.update({
        id: createId(idParts),
        index: this.getIndex(),
        refresh: refresh !== null && refresh !== void 0 ? refresh : false,
        doc: {
          criticality_level: _constants.CRITICALITY_VALUES.DELETED,
          asset: {
            criticality: _constants.CRITICALITY_VALUES.DELETED
          },
          '@timestamp': new Date().toISOString(),
          ...(0, _helpers.getImplicitEntityFields)({
            ...idParts,
            criticalityLevel: _constants.CRITICALITY_VALUES.DELETED
          })
        }
      });
    } catch (err) {
      this.options.logger.error(`Failed to delete asset criticality record: ${err.message}`);
      throw err;
    }
    return record;
  }
  formatSearchResponse(response) {
    var _response$hits$total$, _response$hits$total;
    const records = response.hits.hits.map(hit => hit._source);
    const total = typeof response.hits.total === 'number' ? response.hits.total : (_response$hits$total$ = (_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) !== null && _response$hits$total$ !== void 0 ? _response$hits$total$ : 0;
    return {
      records,
      total
    };
  }
}
exports.AssetCriticalityDataClient = AssetCriticalityDataClient;