"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.objectPairIntersection = exports.objectArrayIntersection = exports.buildAlertRoot = exports.buildAlertGroupFromSequence = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _alert_detail_path = require("../../../../../common/utils/alert_detail_path");
var _constants = require("../../../../../common/constants");
var _build_alert = require("../factories/utils/build_alert");
var _transform_hit_to_alert = require("../factories/utils/transform_hit_to_alert");
var _generate_building_block_ids = require("../factories/utils/generate_building_block_ids");
var _field_names = require("../../../../../common/field_maps/field_names");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// eql shell alerts can have a subAlerts property
// when suppression is used in EQL sequence queries

/**
 * Takes N raw documents from ES that form a sequence and builds them into N+1 signals ready to be indexed -
 * one signal for each event in the sequence, and a "shell" signal that ties them all together. All N+1 signals
 * share the same signal.group.id to make it easy to query them.
 * @param sequence The raw ES documents that make up the sequence
 * @param completeRule object representing the rule that found the sequence
 */
const buildAlertGroupFromSequence = ({
  sharedParams,
  sequence,
  buildReasonMessage
}) => {
  const {
    alertTimestampOverride,
    intendedTimestamp,
    completeRule,
    spaceId,
    inputIndex: indicesToQuery,
    ruleExecutionLogger,
    publicBaseUrl
  } = sharedParams;
  const ancestors = sequence.events.flatMap(event => (0, _build_alert.buildAncestors)(event));
  if (ancestors.some(ancestor => (ancestor === null || ancestor === void 0 ? void 0 : ancestor.rule) === completeRule.alertId)) {
    return {
      shellAlert: undefined,
      buildingBlocks: []
    };
  }

  // The "building block" alerts start out as regular BaseFields.
  // We'll add the group ID and index fields
  // after creating the shell alert later on
  // since that's when the group ID is determined.
  let baseAlerts = [];
  try {
    baseAlerts = sequence.events.map(event => (0, _transform_hit_to_alert.transformHitToAlert)({
      sharedParams,
      doc: event,
      applyOverrides: false,
      buildReasonMessage,
      alertUuid: 'placeholder-alert-uuid' // This is overriden below
    }));
  } catch (error) {
    ruleExecutionLogger.error(error);
    return {
      shellAlert: undefined,
      buildingBlocks: []
    };
  }

  // The ID of each building block alert depends on all of the other building blocks as well,
  // so we generate the IDs after making all the BaseFields
  const buildingBlockIds = (0, _generate_building_block_ids.generateBuildingBlockIds)(baseAlerts);
  const wrappedBaseFields = baseAlerts.map((block, i) => ({
    _id: buildingBlockIds[i],
    _index: '',
    _source: {
      ...block,
      [_ruleDataUtils.ALERT_UUID]: buildingBlockIds[i]
    }
  }));

  // Now that we have an array of building blocks for the events in the sequence,
  // we can build the signal that links the building blocks together
  // and also insert the group id (which is also the "shell" signal _id) in each building block
  const shellAlert = buildAlertRoot({
    wrappedBuildingBlocks: wrappedBaseFields,
    completeRule,
    spaceId,
    buildReasonMessage,
    indicesToQuery,
    alertTimestampOverride,
    publicBaseUrl,
    intendedTimestamp
  });
  const sequenceAlert = {
    _id: shellAlert[_ruleDataUtils.ALERT_UUID],
    _index: '',
    _source: shellAlert
  };

  // Finally, we have the group id from the shell alert so we can convert the BaseFields into EqlBuildingBlocks
  const wrappedBuildingBlocks = wrappedBaseFields.map((block, i) => {
    const alertUrl = (0, _alert_detail_path.getAlertDetailsUrl)({
      alertId: block._id,
      index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
      timestamp: block._source['@timestamp'],
      basePath: publicBaseUrl,
      spaceId
    });
    return {
      ...block,
      _source: {
        ...block._source,
        [_field_names.ALERT_BUILDING_BLOCK_TYPE]: 'default',
        [_field_names.ALERT_GROUP_ID]: shellAlert[_field_names.ALERT_GROUP_ID],
        [_field_names.ALERT_GROUP_INDEX]: i,
        [_ruleDataUtils.ALERT_URL]: alertUrl
      }
    };
  });
  return {
    shellAlert: sequenceAlert,
    buildingBlocks: wrappedBuildingBlocks
  };
};
exports.buildAlertGroupFromSequence = buildAlertGroupFromSequence;
const buildAlertRoot = ({
  wrappedBuildingBlocks,
  completeRule,
  spaceId,
  buildReasonMessage,
  indicesToQuery,
  alertTimestampOverride,
  publicBaseUrl,
  intendedTimestamp
}) => {
  const mergedAlerts = objectArrayIntersection(wrappedBuildingBlocks.map(alert => alert._source));
  const reason = buildReasonMessage({
    name: completeRule.ruleConfig.name,
    severity: completeRule.ruleParams.severity,
    mergedDoc: {
      _source: mergedAlerts
    }
  });
  const doc = (0, _build_alert.buildAlertFields)({
    docs: wrappedBuildingBlocks,
    completeRule,
    spaceId,
    reason,
    indicesToQuery,
    alertUuid: 'placeholder-uuid',
    // These will be overriden below
    publicBaseUrl,
    // Not necessary now, but when the ID is created ahead of time this can be passed
    alertTimestampOverride,
    intendedTimestamp
  });
  const alertId = (0, _build_alert.generateAlertId)(doc);
  const alertUrl = (0, _alert_detail_path.getAlertDetailsUrl)({
    alertId,
    index: `${_constants.DEFAULT_ALERTS_INDEX}-${spaceId}`,
    timestamp: doc['@timestamp'],
    basePath: publicBaseUrl,
    spaceId
  });
  return {
    ...mergedAlerts,
    ...doc,
    [_ruleDataUtils.ALERT_UUID]: alertId,
    [_field_names.ALERT_GROUP_ID]: alertId,
    [_ruleDataUtils.ALERT_URL]: alertUrl
  };
};

/**
 * Merges array of alert sources with the first item in the array
 * @param objects array of alert _source objects
 * @returns singular object
 */
exports.buildAlertRoot = buildAlertRoot;
const objectArrayIntersection = objects => {
  if (objects.length === 0) {
    return undefined;
  } else if (objects.length === 1) {
    return objects[0];
  } else {
    return objects.slice(1).reduce((acc, obj) => objectPairIntersection(acc, obj), objects[0]);
  }
};

/**
 * Finds the intersection of two objects by recursively
 * finding the "intersection" of each of of their common keys'
 * values. If an intersection cannot be found between a key's
 * values, the value will be undefined in the returned object.
 *
 * @param a object
 * @param b object
 * @returns intersection of the two objects
 */
exports.objectArrayIntersection = objectArrayIntersection;
const objectPairIntersection = (a, b) => {
  if (a === undefined || b === undefined) {
    return undefined;
  }
  const intersection = {};
  Object.entries(a).forEach(([key, aVal]) => {
    if (key in b) {
      const bVal = b[key];
      if (typeof aVal === 'object' && !(aVal instanceof Array) && aVal !== null && typeof bVal === 'object' && !(bVal instanceof Array) && bVal !== null) {
        intersection[key] = objectPairIntersection(aVal, bVal);
      } else if (aVal === bVal) {
        intersection[key] = aVal;
      } else if ((0, _lodash.isArray)(aVal) && (0, _lodash.isArray)(bVal)) {
        intersection[key] = (0, _lodash.intersection)(aVal, bVal);
      } else if ((0, _lodash.isArray)(aVal) && !(0, _lodash.isArray)(bVal)) {
        intersection[key] = (0, _lodash.intersection)(aVal, [bVal]);
      } else if (!(0, _lodash.isArray)(aVal) && (0, _lodash.isArray)(bVal)) {
        intersection[key] = (0, _lodash.intersection)([aVal], bVal);
      }
    }
  });
  // Count up the number of entries that are NOT undefined in the intersection
  // If there are no keys OR all entries are undefined, return undefined
  if (Object.values(intersection).reduce((acc, value) => value !== undefined ? acc + 1 : acc, 0) === 0) {
    return undefined;
  } else {
    return intersection;
  }
};
exports.objectPairIntersection = objectPairIntersection;