"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRuleMonitoringService = void 0;
var _server = require("@kbn/core/server");
var _invariant = require("../../../../../common/utils/invariant");
var _with_security_span = require("../../../../utils/with_security_span");
var _client = require("./rule_execution_log/client_for_routes/client");
var _client2 = require("./rule_execution_log/client_for_executors/client");
var _register_event_log_provider = require("./event_log/register_event_log_provider");
var _detection_engine_health_client = require("./detection_engine_health/detection_engine_health_client");
var _event_log_health_client = require("./detection_engine_health/event_log/event_log_health_client");
var _rule_objects_health_client = require("./detection_engine_health/rule_objects/rule_objects_health_client");
var _rule_spaces_client = require("./detection_engine_health/rule_spaces/rule_spaces_client");
var _event_log_reader = require("./rule_execution_log/event_log/event_log_reader");
var _event_log_writer = require("./rule_execution_log/event_log/event_log_writer");
var _fetch_rule_execution_settings = require("./rule_execution_log/execution_settings/fetch_rule_execution_settings");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createRuleMonitoringService = (config, logger) => {
  let coreSetup = null;
  let pluginsSetup = null;
  let coreStart = null;
  return {
    setup: (core, plugins) => {
      coreSetup = core;
      pluginsSetup = plugins;
      (0, _register_event_log_provider.registerEventLogProvider)(plugins.eventLog);
    },
    start: (core, plugins) => {
      coreStart = core;
    },
    createDetectionEngineHealthClient: params => {
      (0, _invariant.invariant)(coreStart, 'Dependencies of RuleMonitoringService are not initialized');
      const {
        rulesClient,
        eventLogClient,
        currentSpaceId
      } = params;
      const {
        savedObjects
      } = coreStart;

      // Create a saved objects client and an importer that can work with saved objects on behalf
      // of the internal Kibana user. This is important because we want to let users with access
      // to only Security Solution to be able to:
      //   1. Install our internal assets like rule monitoring dashboard without the need to
      //      configure the additional `Saved Objects Management: All` privilege.
      //   2. Aggregate rules in all Kibana spaces to get a health overview of the whole cluster -
      //      without having explicit access to every existing space.
      const internalSavedObjectsRepository = savedObjects.createInternalRepository([
      // Note: we include the "alert" hidden SO type here otherwise we would not be able to query it.
      // If at some point it is not considered a hidden type this can be removed.
      'alert']);
      const internalSavedObjectsClient = new _server.SavedObjectsClient(internalSavedObjectsRepository);
      const internalSavedObjectsImporter = savedObjects.createImporter(internalSavedObjectsClient);
      const ruleSpacesClient = (0, _rule_spaces_client.createRuleSpacesClient)(currentSpaceId, internalSavedObjectsClient, logger);
      const ruleObjectsHealthClient = (0, _rule_objects_health_client.createRuleObjectsHealthClient)(rulesClient, internalSavedObjectsClient, logger);
      const eventLogHealthClient = (0, _event_log_health_client.createEventLogHealthClient)(eventLogClient, ruleSpacesClient, logger);
      return (0, _detection_engine_health_client.createDetectionEngineHealthClient)(ruleSpacesClient, ruleObjectsHealthClient, eventLogHealthClient, internalSavedObjectsImporter, logger);
    },
    createRuleExecutionLogClientForRoutes: params => {
      const {
        eventLogClient
      } = params;
      const eventLogReader = (0, _event_log_reader.createEventLogReader)(eventLogClient);
      return (0, _client.createRuleExecutionLogClientForRoutes)(eventLogReader, logger);
    },
    createRuleExecutionLogClientForExecutors: params => {
      return (0, _with_security_span.withSecuritySpan)('IRuleMonitoringService.createRuleExecutionLogClientForExecutors', async () => {
        const {
          savedObjectsClient,
          context,
          ruleMonitoringService,
          ruleResultService
        } = params;
        (0, _invariant.invariant)(coreSetup, 'Dependencies of RuleMonitoringService are not initialized');
        (0, _invariant.invariant)(pluginsSetup, 'Dependencies of RuleMonitoringService are not initialized');
        (0, _invariant.invariant)(ruleMonitoringService, 'ruleMonitoringService required for detection rules');
        (0, _invariant.invariant)(ruleResultService, 'ruleResultService required for detection rules');
        const childLogger = logger.get('ruleExecution');
        const ruleExecutionSettings = await (0, _fetch_rule_execution_settings.fetchRuleExecutionSettings)(config, childLogger, coreSetup, savedObjectsClient);
        const eventLogWriter = (0, _event_log_writer.createEventLogWriter)(pluginsSetup.eventLog);
        return (0, _client2.createRuleExecutionLogClientForExecutors)(ruleExecutionSettings, eventLogWriter, childLogger, context, ruleMonitoringService, ruleResultService);
      });
    }
  };
};
exports.createRuleMonitoringService = createRuleMonitoringService;