"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleHealthRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _rule_monitoring = require("../../../../../../../common/api/detection_engine/rule_monitoring");
var _route_validation = require("../../../../../../utils/build_validation/route_validation");
var _utils = require("../../../../routes/utils");
var _health_timings = require("../health_timings");
var _get_rule_health_request = require("./get_rule_health_request");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get health overview of a rule. Scope: a given detection rule in the current Kibana space.
 * Returns:
 * - health state at the moment of the API call (rule and its execution summary)
 * - health stats over a specified period of time ("health interval")
 * - health stats history within the same interval in the form of a histogram
 *   (the same stats are calculated over each of the discreet sub-intervals of the whole interval)
 */
const getRuleHealthRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _rule_monitoring.GET_RULE_HEALTH_URL,
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_rule_monitoring.GetRuleHealthRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const params = (0, _get_rule_health_request.validateGetRuleHealthRequest)(request.body);
      const ctx = await context.resolve(['securitySolution']);
      const healthClient = ctx.securitySolution.getDetectionEngineHealthClient();
      const ruleHealthParameters = {
        interval: params.interval,
        rule_id: params.ruleId
      };
      const ruleHealth = await healthClient.calculateRuleHealth(ruleHealthParameters);
      const responseBody = {
        timings: (0, _health_timings.calculateHealthTimings)(params.requestReceivedAt),
        parameters: ruleHealthParameters,
        health: {
          ...ruleHealth,
          debug: params.debug ? ruleHealth.debug : undefined
        }
      };
      return response.ok({
        body: responseBody
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.getRuleHealthRoute = getRuleHealthRoute;