"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.sendRuleUpdateTelemetryEvents = sendRuleUpdateTelemetryEvents;
var _prebuilt_rules = require("../../../../../../common/api/detection_engine/prebuilt_rules");
var _events = require("../../../../telemetry/event_based/events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function sendRuleUpdateTelemetryEvents(analytics, RuleUpdateContextsMap, updatedRules, installationErrors, skippedRules) {
  try {
    for (const ruleResponse of updatedRules) {
      const ruleUpdateContext = RuleUpdateContextsMap.get(ruleResponse.rule_id);
      if (ruleUpdateContext) {
        const event = createRuleUpdateTelemetryEvent({
          fieldsDiff: ruleUpdateContext.fieldsDiff,
          ruleId: ruleUpdateContext.ruleId,
          ruleName: ruleUpdateContext.ruleName,
          hasBaseVersion: ruleUpdateContext.hasBaseVersion,
          finalResult: 'SUCCESS'
        });
        analytics.reportEvent(_events.DETECTION_RULE_UPGRADE_EVENT.eventType, event);
      }
    }
    for (const erroredRule of installationErrors) {
      const ruleUpdateContext = RuleUpdateContextsMap.get(erroredRule.item.rule_id);
      if (ruleUpdateContext) {
        const event = createRuleUpdateTelemetryEvent({
          fieldsDiff: ruleUpdateContext.fieldsDiff,
          ruleId: ruleUpdateContext.ruleId,
          ruleName: ruleUpdateContext.ruleName,
          hasBaseVersion: ruleUpdateContext.hasBaseVersion,
          finalResult: 'ERROR'
        });
        analytics.reportEvent(_events.DETECTION_RULE_UPGRADE_EVENT.eventType, event);
      }
    }
    for (const skippedRule of skippedRules) {
      const ruleUpdateContext = RuleUpdateContextsMap.get(skippedRule.rule_id);
      if (ruleUpdateContext) {
        const event = createRuleUpdateTelemetryEvent({
          fieldsDiff: ruleUpdateContext.fieldsDiff,
          ruleId: ruleUpdateContext.ruleId,
          ruleName: ruleUpdateContext.ruleName,
          hasBaseVersion: ruleUpdateContext.hasBaseVersion,
          finalResult: 'SKIP'
        });
        analytics.reportEvent(_events.DETECTION_RULE_UPGRADE_EVENT.eventType, event);
      }
    }
  } catch (e) {
    // we don't want telemetry errors to impact the main flow
    // eslint-disable-next-line no-console
    console.error('Failed to send detection rule update telemetry', e);
  }
}
function createRuleUpdateTelemetryEvent({
  fieldsDiff,
  ruleId,
  ruleName,
  hasBaseVersion,
  finalResult
}) {
  const updatedFieldsTotal = [];
  const updatedFieldsWithNonSolvableConflicts = [];
  const updatedFieldsWithSolvableConflicts = [];
  const updatedFieldsWithNoConflicts = [];
  Object.entries(fieldsDiff).forEach(([fieldName, diff]) => {
    if (fieldName === 'version') {
      return;
    }
    const isUpdatableOutcome = diff.diff_outcome === _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueSameUpdate || diff.diff_outcome === _prebuilt_rules.ThreeWayDiffOutcome.StockValueCanUpdate || diff.diff_outcome === _prebuilt_rules.ThreeWayDiffOutcome.CustomizedValueCanUpdate || diff.diff_outcome === _prebuilt_rules.ThreeWayDiffOutcome.MissingBaseCanUpdate;
    if (!isUpdatableOutcome) {
      return;
    }
    updatedFieldsTotal.push(fieldName);
    switch (diff.conflict) {
      case _prebuilt_rules.ThreeWayDiffConflict.NON_SOLVABLE:
        updatedFieldsWithNonSolvableConflicts.push(fieldName);
        break;
      case _prebuilt_rules.ThreeWayDiffConflict.SOLVABLE:
        updatedFieldsWithSolvableConflicts.push(fieldName);
        break;
      case _prebuilt_rules.ThreeWayDiffConflict.NONE:
      default:
        updatedFieldsWithNoConflicts.push(fieldName);
        break;
    }
  });
  return {
    ruleId,
    ruleName,
    hasBaseVersion,
    updatedFieldsSummary: {
      count: updatedFieldsTotal.length,
      nonSolvableConflictsCount: updatedFieldsWithNonSolvableConflicts.length,
      solvableConflictsCount: updatedFieldsWithSolvableConflicts.length,
      noConflictsCount: updatedFieldsWithNoConflicts.length
    },
    updatedFieldsTotal,
    updatedFieldsWithNonSolvableConflicts,
    updatedFieldsWithSolvableConflicts,
    updatedFieldsWithNoConflicts,
    finalResult
  };
}