"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MicrosoftDefenderEndpointAgentStatusClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/stack-connectors-plugin/common/microsoft_defender_endpoint/constants");
var _lodash = require("lodash");
var _pMap = _interopRequireDefault(require("p-map"));
var _stringify = require("../../../../utils/stringify");
var _errors = require("../errors");
var _ = require("../../..");
var _base_agent_status_client = require("../lib/base_agent_status_client");
var _types = require("../../../../../../common/endpoint/types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MicrosoftDefenderEndpointAgentStatusClient extends _base_agent_status_client.AgentStatusClient {
  constructor(options) {
    super(options);
    (0, _defineProperty2.default)(this, "agentType", 'microsoft_defender_endpoint');
    (0, _defineProperty2.default)(this, "connectorActions", void 0);
    if (!options.connectorActionsClient) {
      throw new _errors.AgentStatusClientError('connectorActionsClient is required to create an instance of MicrosoftDefenderEndpointAgentStatusClient');
    }
    this.connectorActions = new _.NormalizedExternalConnectorClient(options.connectorActionsClient, this.log);
    this.connectorActions.setup(_constants.MICROSOFT_DEFENDER_ENDPOINT_CONNECTOR_ID);
  }
  getAgentStatusFromMachineHealthStatus(healthStatus) {
    // Definition of sensor health status can be found here:
    // https://learn.microsoft.com/en-us/defender-endpoint/check-sensor-status

    switch (healthStatus) {
      case 'Active':
        return _types.HostStatus.HEALTHY;
      case 'Inactive':
        return _types.HostStatus.INACTIVE;
      case 'ImpairedCommunication':
      case 'NoSensorData':
      case 'NoSensorDataImpairedCommunication':
        return _types.HostStatus.UNHEALTHY;
      default:
        return _types.HostStatus.UNENROLLED;
    }
  }
  async calculateHostIsolatedState(agentIds) {
    const response = {};
    const errors = [];
    await (0, _pMap.default)(agentIds, async agentId => {
      response[agentId] = false;
      try {
        // Microsoft's does not seem to have a public API that enables us to get the Isolation state for a machine. To
        // get around this, we query the list of machine actions for each host and look at the last successful
        // Isolate or Unisolate action to determine if host is isolated or not.
        const {
          data: hostLastSuccessfulMachineAction
        } = await this.connectorActions.execute({
          params: {
            subAction: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_ACTIONS,
            subActionParams: {
              status: 'Succeeded',
              type: ['Isolate', 'Unisolate'],
              machineId: agentId,
              pageSize: 1,
              sortField: 'lastUpdateDateTimeUtc',
              sortDirection: 'desc'
            }
          }
        });

        // Only check the first action if the array is not empty
        const [firstAction] = (hostLastSuccessfulMachineAction === null || hostLastSuccessfulMachineAction === void 0 ? void 0 : hostLastSuccessfulMachineAction.value) || [];
        if ((firstAction === null || firstAction === void 0 ? void 0 : firstAction.type) === 'Isolate') {
          response[agentId] = true;
        }
      } catch (err) {
        errors.push(err.message);
      }
    }, {
      concurrency: 2
    });
    if (errors.length > 0) {
      this.log.error(`Attempt to calculate isolate state for Microsoft Defender hosts generated the following errors:\n${errors.join('\n')}`);
    }
    this.log.debug(() => `Microsoft agents isolated state:\n${(0, _stringify.stringify)(response)}`);
    return response;
  }
  async getAgentStatuses(agentIds) {
    const esClient = this.options.esClient;
    const metadataService = this.options.endpointService.getEndpointMetadataService();
    try {
      var _msMachineListRespons;
      const [{
        data: msMachineListResponse
      }, agentIsolationState, allPendingActions] = await Promise.all([this.connectorActions.execute({
        params: {
          subAction: _constants.MICROSOFT_DEFENDER_ENDPOINT_SUB_ACTION.GET_AGENT_LIST,
          subActionParams: {
            id: agentIds
          }
        }
      }),
      // Calculate host's current isolation state
      this.calculateHostIsolatedState(agentIds),
      // Fetch pending actions summary
      (0, _.getPendingActionsSummary)(esClient, metadataService, this.log, agentIds)]);
      const machinesById = (0, _lodash.keyBy)((_msMachineListRespons = msMachineListResponse === null || msMachineListResponse === void 0 ? void 0 : msMachineListResponse.value) !== null && _msMachineListRespons !== void 0 ? _msMachineListRespons : [], 'id');
      const pendingActionsByAgentId = (0, _lodash.keyBy)(allPendingActions, 'agent_id');
      return agentIds.reduce((acc, agentId) => {
        var _agentIsolationState$, _thisMachine$lastSeen, _thisAgentPendingActi;
        const thisMachine = machinesById[agentId];
        const thisAgentPendingActions = pendingActionsByAgentId[agentId];
        acc[agentId] = {
          agentId,
          agentType: this.agentType,
          found: !!thisMachine,
          isolated: (_agentIsolationState$ = agentIsolationState[agentId]) !== null && _agentIsolationState$ !== void 0 ? _agentIsolationState$ : false,
          lastSeen: (_thisMachine$lastSeen = thisMachine === null || thisMachine === void 0 ? void 0 : thisMachine.lastSeen) !== null && _thisMachine$lastSeen !== void 0 ? _thisMachine$lastSeen : '',
          status: this.getAgentStatusFromMachineHealthStatus(thisMachine === null || thisMachine === void 0 ? void 0 : thisMachine.healthStatus),
          pendingActions: (_thisAgentPendingActi = thisAgentPendingActions === null || thisAgentPendingActions === void 0 ? void 0 : thisAgentPendingActions.pending_actions) !== null && _thisAgentPendingActi !== void 0 ? _thisAgentPendingActi : {}
        };
        return acc;
      }, {});
    } catch (err) {
      const error = new _errors.AgentStatusClientError(`Failed to fetch Microsoft Defender for Endpoint agent status for agentIds: [${agentIds}], failed with: ${err.message}`, 500, err);
      this.log.error(error);
      throw error;
    }
  }
}
exports.MicrosoftDefenderEndpointAgentStatusClient = MicrosoftDefenderEndpointAgentStatusClient;