"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TelemetryConfigWatcher = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/fleet-plugin/common");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _policy = require("../../../../common/endpoint/service/policy");
var _stringify = require("../../utils/stringify");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TelemetryConfigWatcher {
  constructor(policyService, esStart, endpointAppContextService, options = {
    immediateRetry: false
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "policyService", void 0);
    (0, _defineProperty2.default)(this, "endpointAppContextService", void 0);
    (0, _defineProperty2.default)(this, "subscription", void 0);
    (0, _defineProperty2.default)(this, "retryOptions", void 0);
    this.policyService = policyService;
    this.esClient = esStart.client.asInternalUser;
    this.endpointAppContextService = endpointAppContextService;
    this.logger = endpointAppContextService.createLogger(this.constructor.name);
    this.retryOptions = {
      retries: 4,
      minTimeout: options.immediateRetry ? 0 : 1000
    };
  }

  /**
   * The policy watcher is not called as part of a HTTP request chain, where the
   * request-scoped SOClient could be passed down. It is called via telemetry observable
   * changes. We are acting as the 'system' in response to telemetry changes, so we are
   * intentionally using the system user here. Be very aware of what you are using this
   * client to do
   */
  makeInternalSOClient() {
    return this.endpointAppContextService.savedObjects.createInternalUnscopedSoClient(false);
  }
  start(telemetryConfigProvider) {
    var _telemetryConfigProvi;
    this.subscription = (_telemetryConfigProvi = telemetryConfigProvider.getObservable()) === null || _telemetryConfigProvi === void 0 ? void 0 : _telemetryConfigProvi.subscribe(this.watch.bind(this));
  }
  stop() {
    if (this.subscription) {
      this.subscription.unsubscribe();
    }
  }
  async watch(isTelemetryEnabled) {
    let page = 1;
    let response;
    let updated = 0;
    let failed = 0;
    this.logger.debug(`Checking Endpoint policies to update due to changed global telemetry config setting. (New value: ${isTelemetryEnabled})`);
    do {
      try {
        response = await (0, _pRetry.default)(() => this.policyService.list(this.makeInternalSOClient(), {
          page,
          perPage: 100,
          kuery: `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name: endpoint`
        }), {
          onFailedAttempt: error => this.logger.debug(`Failed to read package policies on ${error.attemptNumber}. attempt on page ${page}, reason: ${(0, _stringify.stringify)(error)}`),
          ...this.retryOptions
        });
      } catch (e) {
        this.logger.warn(`Unable to verify endpoint policies in line with telemetry change: failed to fetch package policies: ${(0, _stringify.stringify)(e)}`);
        return;
      }
      const updates = [];
      for (const policy of response.items) {
        const updatePolicy = (0, _policy.getPolicyDataForUpdate)(policy);
        const policyConfig = updatePolicy.inputs[0].config.policy.value;
        if (isTelemetryEnabled !== policyConfig.global_telemetry_enabled) {
          policyConfig.global_telemetry_enabled = isTelemetryEnabled;
          updates.push({
            ...updatePolicy,
            id: policy.id
          });
        }
      }
      if (updates.length) {
        try {
          var _updateResult$updated, _updateResult$updated2;
          const updateResult = await (0, _pRetry.default)(() => this.policyService.bulkUpdate(this.makeInternalSOClient(), this.esClient, updates), {
            onFailedAttempt: error => this.logger.debug(`Failed to bulk update package policies on ${error.attemptNumber}. attempt, reason: ${(0, _stringify.stringify)(error)}`),
            ...this.retryOptions
          });
          if (updateResult.failedPolicies.length) {
            this.logger.warn(`Cannot update telemetry flag in the following policies:\n${updateResult.failedPolicies.map(entry => `- id: ${entry.packagePolicy.id}, error: ${(0, _stringify.stringify)(entry.error)}`).join('\n')}`);
          }
          updated += (_updateResult$updated = (_updateResult$updated2 = updateResult.updatedPolicies) === null || _updateResult$updated2 === void 0 ? void 0 : _updateResult$updated2.length) !== null && _updateResult$updated !== void 0 ? _updateResult$updated : 0;
          failed += updateResult.failedPolicies.length;
        } catch (e) {
          this.logger.warn(`Unable to update telemetry config state to ${isTelemetryEnabled} in policies: ${updates.map(update => update.id)}\n\n${(0, _stringify.stringify)(e)}`);
          failed += updates.length;
        }
      }
      page++;
    } while (response.page * response.perPage < response.total);
    this.logger.info(`Finished updating global_telemetry_enabled flag to ${isTelemetryEnabled} in Defend package policies: ${updated} succeeded, ${failed} failed.`);
  }
}
exports.TelemetryConfigWatcher = TelemetryConfigWatcher;