"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.toolDetails = exports.getInspectIndexMappingTool = void 0;
var _tools = require("@langchain/core/tools");
var _zod = require("@kbn/zod");
var _server = require("@kbn/data-views-plugin/server");
var _inspect_index_utils = require("./inspect_index_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const toolDetails = exports.toolDetails = {
  name: 'inspect_index_mapping',
  description: 'Use this tool when there is a "verification_exception Unknown column" error or to see which fields and types are used in the index.' + 'This function will return as much of the index mapping as possible. If the index mapping is too large, some values will be truncated as indicated by "...".' + 'Call the function again to inspected truncated values.' + `Example:
Index mapping:
{"user":{"address":{"city":{"name":{"type":"keyword"},"zip":{"type":"integer"}}}}}}

Function Call #1:
Property: "." // to get the root of the index mapping
Function response: {"user":{"address":{"city":"...","zip":"..."}}}

Function Call #2:
Property: "user.address.city"
Function response: {"name":{"type":"keyword"}
`
};
const getInspectIndexMappingTool = ({
  esClient,
  indexPattern
}) => {
  const indexPatternsFetcher = new _server.IndexPatternsFetcher(esClient);
  return (0, _tools.tool)(async ({
    property
  }) => {
    const {
      fields
    } = await indexPatternsFetcher.getFieldsForWildcard({
      pattern: indexPattern,
      fieldCapsOptions: {
        allow_no_indices: false,
        includeUnmapped: false
      }
    });
    const prunedFields = fields.map(p => ({
      name: p.name,
      type: p.esTypes[0]
    }));
    const nestedObject = (0, _inspect_index_utils.mapFieldDescriptorToNestedObject)(prunedFields);
    const nestedValue = (0, _inspect_index_utils.getNestedValue)(nestedObject, property);
    const result = (0, _inspect_index_utils.shallowObjectViewTruncated)(nestedValue, 30000);
    return result ? JSON.stringify(result) : `No value found for property "${property}".`;
  }, {
    name: toolDetails.name,
    description: toolDetails.description,
    schema: _zod.z.object({
      property: _zod.z.string().default('.').describe(`The property to inspect. The property should be a dot-separated path to the field in the index mapping. For example, "user.name" or "user.address.city". Pass "." to get the root.`)
    })
  });
};
exports.getInspectIndexMappingTool = getInspectIndexMappingTool;