"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ensureTimelineIsSaved = ensureTimelineIsSaved;
exports.refreshTimelines = refreshTimelines;
var _all = require("../../containers/all");
var _inputs = require("../../../common/store/inputs");
var _actions = require("../actions");
var _timeline = require("../../../../common/api/timeline");
var _selectors = require("../selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Refreshes all timelines, so changes are propagated to everywhere on the page
 */
function refreshTimelines(state) {
  const allTimelineQuery = _inputs.inputsSelectors.globalQueryByIdSelector()(state, _all.ALL_TIMELINE_QUERY_ID);
  if (allTimelineQuery.refetch != null) {
    allTimelineQuery.refetch();
  }
}

/**
 * Given a timeline model, it will return that model when the timeline has been saved before,
 * or save a draft version of that timeline.
 * This is a usefull check for when you're working with timeline-associated saved objects
 * which require the exitence of a timeline's `savedObjectId`.
 */
async function ensureTimelineIsSaved({
  localTimelineId,
  timeline,
  store
}) {
  // In case `savedObjectId` exists, the timeline has been saved before.
  if (timeline.savedObjectId) {
    return timeline;
  }

  // The timeline hasn't been saved, so let's create make it a draft.
  await store.dispatch((0, _actions.updateTimeline)({
    id: localTimelineId,
    timeline: {
      ...timeline,
      status: _timeline.TimelineStatusEnum.draft
    }
  }));

  // The draft needs to be persisted
  await store.dispatch((0, _actions.saveTimeline)({
    id: localTimelineId,
    saveAsNew: false
  }));

  // Make sure we're returning the most updated version of the timeline
  return (0, _selectors.selectTimelineById)(store.getState(), localTimelineId);
}