"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TASK_STATS_POLLING_SLEEP_SECONDS = exports.SiemRulesMigrationsService = exports.START_STOP_POLLING_SLEEP_SECONDS = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _constants = require("@kbn/elastic-assistant/impl/assistant_context/constants");
var _lodash = require("lodash");
var _constants2 = require("../../../../common/siem_migrations/constants");
var _experimental_features_service = require("../../../common/experimental_features_service");
var _use_license = require("../../../common/hooks/use_license");
var api = _interopRequireWildcard(require("../api"));
var _capabilities = require("./capabilities");
var _success_notification = require("./notifications/success_notification");
var _storage = require("./storage");
var i18n = _interopRequireWildcard(require("./translations"));
var _telemetry = require("./telemetry");
var _no_connector_notification = require("./notifications/no_connector_notification");
var _missing_capabilities_notification = require("./notifications/missing_capabilities_notification");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// use the default assistant namespace since it's the only one we use
const NAMESPACE_TRACE_OPTIONS_SESSION_STORAGE_KEY = `${_constants.DEFAULT_ASSISTANT_NAMESPACE}.${_constants.TRACE_OPTIONS_SESSION_STORAGE_KEY}`;
const TASK_STATS_POLLING_SLEEP_SECONDS = exports.TASK_STATS_POLLING_SLEEP_SECONDS = 10;
const START_STOP_POLLING_SLEEP_SECONDS = exports.START_STOP_POLLING_SLEEP_SECONDS = 1;
const CREATE_MIGRATION_BODY_BATCH_SIZE = 50;
class SiemRulesMigrationsService {
  constructor(core, plugins, telemetryService) {
    (0, _defineProperty2.default)(this, "latestStats$", void 0);
    (0, _defineProperty2.default)(this, "isPolling", false);
    (0, _defineProperty2.default)(this, "connectorIdStorage", new _storage.RuleMigrationsStorage('connectorId'));
    (0, _defineProperty2.default)(this, "traceOptionsStorage", new _storage.RuleMigrationsStorage('traceOptions', {
      customKey: NAMESPACE_TRACE_OPTIONS_SESSION_STORAGE_KEY,
      storageType: 'session'
    }));
    (0, _defineProperty2.default)(this, "telemetry", void 0);
    this.core = core;
    this.plugins = plugins;
    this.telemetry = new _telemetry.SiemRulesMigrationsTelemetry(telemetryService);
    this.latestStats$ = new _rxjs.BehaviorSubject(null);
    this.plugins.spaces.getActiveSpace().then(space => {
      this.connectorIdStorage.setSpaceId(space.id);
      this.startPolling();
    });
  }

  /** Accessor for the rule migrations API client */
  get api() {
    return api;
  }

  /** Returns the latest stats observable, which is updated every time the stats are fetched */
  getLatestStats$() {
    return this.latestStats$.asObservable().pipe((0, _rxjs.distinctUntilChanged)(_lodash.isEqual));
  }

  /** Returns any missing capabilities for the user to use this feature */
  getMissingCapabilities(level) {
    return (0, _capabilities.getMissingCapabilities)(this.core.application.capabilities, level);
  }

  /** Checks if the user has any missing capabilities for this feature */
  hasMissingCapabilities(level) {
    return this.getMissingCapabilities(level).length > 0;
  }

  /** Checks if the service is available based on the `license`, `capabilities` and `experimentalFeatures` */
  isAvailable() {
    return !_experimental_features_service.ExperimentalFeaturesService.get().siemMigrationsDisabled && _use_license.licenseService.isEnterprise() && !this.hasMissingCapabilities('minimum');
  }

  /** Starts polling the rule migrations stats if not already polling and if the feature is available to the user */
  startPolling() {
    if (this.isPolling || !this.isAvailable()) {
      return;
    }
    this.isPolling = true;
    this.startTaskStatsPolling().catch(e => {
      this.core.notifications.toasts.addError(e, {
        title: i18n.POLLING_ERROR
      });
    }).finally(() => {
      this.isPolling = false;
    });
  }

  /** Adds rules to a rule migration, batching the requests to avoid hitting the max payload size limit of the API */
  async addRulesToMigration(migrationId, rules) {
    const rulesCount = rules.length;
    if (rulesCount === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }

    // Batching creation to avoid hitting the max payload size limit of the API
    for (let i = 0; i < rulesCount; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
      const rulesBatch = rules.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
      await api.addRulesToMigration({
        migrationId,
        body: rulesBatch
      });
    }
  }

  /** Creates a rule migration with a name and adds the rules to it, returning the migration ID */
  async createRuleMigration(data, migrationName) {
    const rulesCount = data.length;
    if (rulesCount === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }
    try {
      // create the migration
      const {
        migration_id: migrationId
      } = await api.createRuleMigration({
        name: migrationName
      });
      await this.addRulesToMigration(migrationId, data);
      this.telemetry.reportSetupMigrationCreated({
        migrationId,
        rulesCount
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationCreated({
        rulesCount,
        error
      });
      throw error;
    }
  }

  /** Deletes a rule migration by its ID, refreshing the stats to remove it from the list */
  async deleteMigration(migrationId) {
    try {
      await api.deleteMigration({
        migrationId
      });

      // Refresh stats to remove the deleted migration from the list. All UI observables will be updated automatically
      await this.getRuleMigrationsStats();
      this.telemetry.reportSetupMigrationDeleted({
        migrationId
      });
      return migrationId;
    } catch (error) {
      this.telemetry.reportSetupMigrationDeleted({
        migrationId,
        error
      });
      throw error;
    }
  }

  /** Upserts resources for a rule migration, batching the requests to avoid hitting the max payload size limit of the API */
  async upsertMigrationResources(migrationId, body) {
    const count = body.length;
    if (count === 0) {
      throw new Error(i18n.EMPTY_RULES_ERROR);
    }
    // We assume all resources are of the same type. There is no use case for mixing types in a single upload
    const type = body[0].type;
    try {
      // Batching creation to avoid hitting the max payload size limit of the API
      for (let i = 0; i < count; i += CREATE_MIGRATION_BODY_BATCH_SIZE) {
        const bodyBatch = body.slice(i, i + CREATE_MIGRATION_BODY_BATCH_SIZE);
        await api.upsertMigrationResources({
          migrationId,
          body: bodyBatch
        });
      }
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count
      });
    } catch (error) {
      this.telemetry.reportSetupResourceUploaded({
        migrationId,
        type,
        count,
        error
      });
      throw error;
    }
  }

  /** Starts a rule migration task and waits for the task to start running */
  async startRuleMigration(migrationId, retry, settings) {
    var _settings$connectorId;
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _missing_capabilities_notification.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        started: false
      };
    }
    const connectorId = (_settings$connectorId = settings === null || settings === void 0 ? void 0 : settings.connectorId) !== null && _settings$connectorId !== void 0 ? _settings$connectorId : this.connectorIdStorage.get();
    const skipPrebuiltRulesMatching = settings === null || settings === void 0 ? void 0 : settings.skipPrebuiltRulesMatching;
    if (!connectorId) {
      this.core.notifications.toasts.add((0, _no_connector_notification.getNoConnectorToast)(this.core));
      return {
        started: false
      };
    }
    const params = {
      migrationId,
      settings: {
        connectorId,
        skipPrebuiltRulesMatching
      },
      retry
    };
    const traceOptions = this.traceOptionsStorage.get();
    if (traceOptions) {
      params.langSmithOptions = {
        project_name: traceOptions.langSmithProject,
        api_key: traceOptions.langSmithApiKey
      };
    }
    try {
      const result = await api.startRuleMigration(params);

      // Should take a few seconds to start the task, so we poll until it is running
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status === _constants2.SiemMigrationTaskStatus.RUNNING, {
        sleepSecs: START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to start
      );
      this.startPolling();
      this.telemetry.reportStartTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStartTranslation({
        ...params,
        error
      });
      throw error;
    }
  }

  /** Stops a running rule migration task and waits for the task to completely stop */
  async stopRuleMigration(migrationId) {
    const missingCapabilities = this.getMissingCapabilities('all');
    if (missingCapabilities.length > 0) {
      this.core.notifications.toasts.add((0, _missing_capabilities_notification.getMissingCapabilitiesToast)(missingCapabilities, this.core));
      return {
        stopped: false
      };
    }
    const params = {
      migrationId
    };
    try {
      const result = await api.stopRuleMigration(params);

      // Should take a few seconds to stop the task, so we poll until it is not running anymore
      await this.migrationTaskPollingUntil(migrationId, ({
        status
      }) => status !== _constants2.SiemMigrationTaskStatus.RUNNING,
      // may be STOPPED, FINISHED or INTERRUPTED
      {
        sleepSecs: START_STOP_POLLING_SLEEP_SECONDS,
        timeoutSecs: 90
      } // wait up to 90 seconds for the task to stop
      );
      this.telemetry.reportStopTranslation(params);
      return result;
    } catch (error) {
      this.telemetry.reportStopTranslation({
        ...params,
        error
      });
      throw error;
    }
  }

  /** Gets the rule migrations stats, retrying on network errors or 503 status */
  async getRuleMigrationsStats(params = {}) {
    const allStats = await this.getRuleMigrationsStatsWithRetry(params);
    this.latestStats$.next(allStats); // Keep the latest stats observable in sync
    return allStats;
  }
  sleep(seconds) {
    return new Promise(resolve => setTimeout(resolve, seconds * 1000));
  }

  /** Polls the migration task stats until the finish condition is met or the timeout is reached. */
  async migrationTaskPollingUntil(migrationId, finishCondition, {
    sleepSecs = 1,
    timeoutSecs = 60
  } = {}) {
    const timeoutId = setTimeout(() => {
      throw new Error('Migration task polling timed out');
    }, timeoutSecs * 1000);
    let retry = true;
    do {
      const stats = await api.getRuleMigrationStats({
        migrationId
      });
      if (finishCondition(stats)) {
        clearTimeout(timeoutId);
        retry = false;
      } else {
        await this.sleep(sleepSecs);
      }
    } while (retry);
    // updates the latest stats observable for all migrations to make sure they are in sync
    await this.getRuleMigrationsStats();
  }

  /** Retries the API call to get rule migrations stats in case of network errors or 503 status */
  async getRuleMigrationsStatsWithRetry(params = {}, sleepSecs) {
    if (sleepSecs) {
      await this.sleep(sleepSecs);
    }
    return api.getRuleMigrationsStatsAll(params).catch(e => {
      var _e$response;
      // Retry only on network errors (no status) and 503 (Service Unavailable), otherwise throw
      const status = ((_e$response = e.response) === null || _e$response === void 0 ? void 0 : _e$response.status) || e.status;
      if (status && status !== 503) {
        throw e;
      }
      const nextSleepSecs = sleepSecs ? sleepSecs * 2 : 1; // Exponential backoff
      if (nextSleepSecs > 60) {
        // Wait for a minutes max (two minutes total) for the API to be available again
        throw e;
      }
      return this.getRuleMigrationsStatsWithRetry(params, nextSleepSecs);
    });
  }

  /** Starts polling the rule migrations stats and handles the notifications for finished migrations */
  async startTaskStatsPolling() {
    let pendingMigrationIds = [];
    do {
      const results = await this.getRuleMigrationsStats();
      if (pendingMigrationIds.length > 0) {
        // send notifications for finished migrations
        pendingMigrationIds.forEach(pendingMigrationId => {
          const migrationStats = results.find(item => item.id === pendingMigrationId);
          if ((migrationStats === null || migrationStats === void 0 ? void 0 : migrationStats.status) === _constants2.SiemMigrationTaskStatus.FINISHED) {
            this.core.notifications.toasts.addSuccess((0, _success_notification.getSuccessToast)(migrationStats, this.core));
          }
        });
      }

      // reprocess pending migrations
      pendingMigrationIds = [];
      for (const result of results) {
        var _result$last_executio;
        if (result.status === _constants2.SiemMigrationTaskStatus.RUNNING) {
          pendingMigrationIds.push(result.id);
        }

        // automatically resume interrupted migrations when the proper conditions are met
        if (result.status === _constants2.SiemMigrationTaskStatus.INTERRUPTED && !((_result$last_executio = result.last_execution) !== null && _result$last_executio !== void 0 && _result$last_executio.error)) {
          var _result$last_executio2, _result$last_executio3, _result$last_executio4;
          const connectorId = (_result$last_executio2 = (_result$last_executio3 = result.last_execution) === null || _result$last_executio3 === void 0 ? void 0 : _result$last_executio3.connector_id) !== null && _result$last_executio2 !== void 0 ? _result$last_executio2 : this.connectorIdStorage.get();
          const skipPrebuiltRulesMatching = (_result$last_executio4 = result.last_execution) === null || _result$last_executio4 === void 0 ? void 0 : _result$last_executio4.skip_prebuilt_rules_matching;
          if (connectorId && !this.hasMissingCapabilities('all')) {
            await api.startRuleMigration({
              migrationId: result.id,
              settings: {
                connectorId,
                skipPrebuiltRulesMatching
              }
            });
            pendingMigrationIds.push(result.id);
          }
        }
      }

      // Do not wait if there are no more pending migrations
      if (pendingMigrationIds.length > 0) {
        await this.sleep(TASK_STATS_POLLING_SLEEP_SECONDS);
      }
    } while (pendingMigrationIds.length > 0);
  }
}
exports.SiemRulesMigrationsService = SiemRulesMigrationsService;