"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.timelineDataToEnrichment = exports.parseExistingEnrichments = exports.isInvestigationTimeEnrichment = exports.getShimmedIndicatorValue = exports.getFirstSeen = exports.getEnrichmentValue = exports.getEnrichmentIdentifiers = exports.getEnrichmentFields = exports.filterDuplicateEnrichments = exports.buildThreatDetailsItems = void 0;
var _lodash = require("lodash");
var _common = require("@kbn/timelines-plugin/common");
var _i18n = require("@kbn/i18n");
var _search_strategy = require("../../../../../common/search_strategy");
var _constants = require("../../../../../common/constants");
var _constants2 = require("../../../../../common/cti/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NESTED_OBJECT_VALUES_NOT_RENDERED = _i18n.i18n.translate('xpack.securitySolution.flyout.threatIntelligence.investigationEnrichmentObjectValuesNotRendered', {
  defaultMessage: 'This field contains nested object values, which are not rendered here. See the full document for all fields/values'
});

/**
 *  Retrieves the first element of the given array.
 *
 * @param array the array to retrieve a value from
 * @returns the first element of the array, or undefined if the array is undefined
 */
const getFirstElement = array => array ? array[0] : undefined;

/**
 * Returns true if the enrichment type is 'investigation_time'
 */
const isInvestigationTimeEnrichment = type => type === _constants2.ENRICHMENT_TYPES.InvestigationTime;

/**
 * Parses existing enrichments from the timeline data
 */
exports.isInvestigationTimeEnrichment = isInvestigationTimeEnrichment;
const parseExistingEnrichments = data => {
  const threatIndicatorField = data.find(({
    field,
    originalValue
  }) => field === _constants.ENRICHMENT_DESTINATION_PATH && originalValue);
  if (!threatIndicatorField) {
    return [];
  }
  const {
    originalValue
  } = threatIndicatorField;
  const enrichmentStrings = Array.isArray(originalValue) ? originalValue : [originalValue];
  return enrichmentStrings.reduce((enrichments, enrichmentString) => {
    try {
      const enrichment = (0, _common.getDataFromFieldsHits)(JSON.parse(enrichmentString));
      enrichments.push(enrichment);
    } catch (e) {
      // omit failed parse
    }
    return enrichments;
  }, []);
};

/**
 * Converts timeline data to a CtiEnrichment object
 */
exports.parseExistingEnrichments = parseExistingEnrichments;
const timelineDataToEnrichment = data => data.reduce((acc, item) => {
  acc[item.field] = item.originalValue;
  return acc;
}, {});

/**
 * Extracts the first value from an enrichment field
 */
exports.timelineDataToEnrichment = timelineDataToEnrichment;
const getEnrichmentValue = (enrichment, field) => getFirstElement(enrichment[field]);

/**
 * These fields (e.g. 'indicator.ip') may be in one of three places depending on whether it's:
 *   * a queried, legacy filebeat indicator ('threatintel.indicator.ip')
 *   * a queried, ECS 1.11 filebeat indicator ('threat.indicator.ip')
 *   * an existing indicator from an enriched alert ('indicator.ip')
 */
exports.getEnrichmentValue = getEnrichmentValue;
const getShimmedIndicatorValue = (enrichment, field) => getEnrichmentValue(enrichment, field) || getEnrichmentValue(enrichment, `threatintel.${field}`) || getEnrichmentValue(enrichment, `threat.${field}`);

/**
 * Extracts the identifiers from an enrichment
 */
exports.getShimmedIndicatorValue = getShimmedIndicatorValue;
const getEnrichmentIdentifiers = enrichment => ({
  id: getEnrichmentValue(enrichment, _constants2.MATCHED_ID),
  field: getEnrichmentValue(enrichment, _constants2.MATCHED_FIELD),
  value: getEnrichmentValue(enrichment, _constants2.MATCHED_ATOMIC),
  type: getEnrichmentValue(enrichment, _constants2.MATCHED_TYPE),
  feedName: getShimmedIndicatorValue(enrichment, _constants2.FEED_NAME)
});

/**
 * Returns a string composed of the id and the field for the enrichment
 */
exports.getEnrichmentIdentifiers = getEnrichmentIdentifiers;
const buildEnrichmentId = enrichment => {
  const {
    id,
    field
  } = getEnrichmentIdentifiers(enrichment);
  return `${id}${field}`;
};

/**
 * This function receives an array of enrichments and removes
 * investigation-time enrichments if that exact indicator already exists
 * elsewhere in the list.
 *
 * @param enrichments {@type CtiEnrichment[]}
 */
const filterDuplicateEnrichments = enrichments => {
  if (enrichments.length < 2) {
    return enrichments;
  }
  const enrichmentsById = (0, _lodash.groupBy)(enrichments, buildEnrichmentId);
  return Object.values(enrichmentsById).map(enrichmentGroup => {
    var _enrichmentGroup$find;
    return (_enrichmentGroup$find = enrichmentGroup.find(enrichment => !isInvestigationTimeEnrichment(getEnrichmentValue(enrichment, _constants2.MATCHED_TYPE)))) !== null && _enrichmentGroup$find !== void 0 ? _enrichmentGroup$find : enrichmentGroup[0];
  });
};

/**
 * Returns the fields from the enrichments
 */
exports.filterDuplicateEnrichments = filterDuplicateEnrichments;
const getEnrichmentFields = items => items.reduce((fields, item) => {
  if ((0, _search_strategy.isValidEventField)(item.field)) {
    const value = getFirstElement(item.originalValue);
    if (value) {
      return {
        ...fields,
        [item.field]: value
      };
    }
  }
  return fields;
}, {});

/**
 * Returns the first seen date from the enrichment
 */
exports.getEnrichmentFields = getEnrichmentFields;
const getFirstSeen = enrichment => {
  const firstSeenValue = getShimmedIndicatorValue(enrichment, _constants2.FIRST_SEEN);
  const firstSeenDate = Date.parse(firstSeenValue !== null && firstSeenValue !== void 0 ? firstSeenValue : 'no date');
  return Number.isInteger(firstSeenDate) ? firstSeenDate : new Date(-1).valueOf();
};

/**
 * Builds the threat details items for the summary table
 */
exports.getFirstSeen = getFirstSeen;
const buildThreatDetailsItems = enrichment => Object.keys(enrichment).sort().map(field => {
  const title = field.startsWith(_constants.DEFAULT_INDICATOR_SOURCE_PATH) ? field.replace(`${_constants.DEFAULT_INDICATOR_SOURCE_PATH}`, 'indicator') : field;
  let value = getFirstElement(enrichment[field]);
  if ((0, _lodash.isObject)(value)) {
    value = NESTED_OBJECT_VALUES_NOT_RENDERED;
  }
  return {
    title,
    description: {
      fieldName: field,
      value: value
    }
  };
});
exports.buildThreatDetailsItems = buildThreatDetailsItems;