"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAlertSummary = void 0;
var _react = require("react");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _elasticAssistant = require("@kbn/elastic-assistant");
var _use_fetch_alert_summary = require("./use_fetch_alert_summary");
var _use_bulk_update_alert_summary = require("./use_bulk_update_alert_summary");
var i18n = _interopRequireWildcard(require("../constants/translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Hook that generates the alert AI summary along side other related items
 */
const useAlertSummary = ({
  alertId,
  defaultConnectorId,
  promptContext,
  showAnonymizedValues = false
}) => {
  const {
    abortStream,
    sendMessage
  } = (0, _elasticAssistant.useChatComplete)({
    connectorId: defaultConnectorId
  });
  const {
    data: anonymizationFields,
    isFetched: isFetchedAnonymizationFields
  } = (0, _elasticAssistant.useFetchAnonymizationFields)();
  const [isConnectorMissing, setIsConnectorMissing] = (0, _react.useState)(false);
  const [alertSummary, setAlertSummary] = (0, _react.useState)(i18n.NO_SUMMARY_AVAILABLE);
  const [recommendedActions, setRecommendedActions] = (0, _react.useState)();
  const [messageAndReplacements, setMessageAndReplacements] = (0, _react.useState)(null);
  // indicates that an alert summary exists or is being created/fetched
  const [hasAlertSummary, setHasAlertSummary] = (0, _react.useState)(false);
  const {
    data: fetchedAlertSummary,
    refetch: refetchAlertSummary,
    isFetched: isAlertSummaryFetched
  } = (0, _use_fetch_alert_summary.useFetchAlertSummary)({
    alertId,
    connectorId: defaultConnectorId
  });
  const {
    bulkUpdate
  } = (0, _use_bulk_update_alert_summary.useBulkUpdateAlertSummary)();
  (0, _react.useEffect)(() => {
    if (fetchedAlertSummary.data.length > 0) {
      setHasAlertSummary(true);
      setAlertSummary(showAnonymizedValues ? fetchedAlertSummary.data[0].summary : (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
        messageContent: fetchedAlertSummary.data[0].summary,
        replacements: fetchedAlertSummary.data[0].replacements
      }));
      if (fetchedAlertSummary.data[0].recommendedActions) {
        setRecommendedActions(showAnonymizedValues ? fetchedAlertSummary.data[0].recommendedActions : (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
          messageContent: fetchedAlertSummary.data[0].recommendedActions,
          replacements: fetchedAlertSummary.data[0].replacements
        }));
      }
    }
  }, [fetchedAlertSummary, showAnonymizedValues]);
  (0, _react.useEffect)(() => {
    const fetchContext = async () => {
      var _userMessage$replacem, _userMessage$content;
      const newSelectedPromptContext = await (0, _elasticAssistant.getNewSelectedPromptContext)({
        anonymizationFields,
        promptContext
      });
      const selectedPromptContexts = {
        [promptContext.id]: newSelectedPromptContext
      };
      const userMessage = (0, _elasticAssistant.getCombinedMessage)({
        currentReplacements: {},
        promptText: fetchedAlertSummary.prompt,
        selectedPromptContexts
      });
      const baseReplacements = (_userMessage$replacem = userMessage.replacements) !== null && _userMessage$replacem !== void 0 ? _userMessage$replacem : {};
      const selectedPromptContextsReplacements = Object.values(selectedPromptContexts).reduce((acc, context) => ({
        ...acc,
        ...context.replacements
      }), {});
      const replacements = {
        ...baseReplacements,
        ...selectedPromptContextsReplacements
      };
      setMessageAndReplacements({
        message: (_userMessage$content = userMessage.content) !== null && _userMessage$content !== void 0 ? _userMessage$content : '',
        replacements
      });
    };
    if (isFetchedAnonymizationFields && isAlertSummaryFetched) fetchContext();
  }, [anonymizationFields, isFetchedAnonymizationFields, isAlertSummaryFetched, fetchedAlertSummary.prompt, promptContext]);
  const [isGenerating, setIsGenerating] = (0, _react.useState)(false);
  const fetchAISummary = (0, _react.useCallback)(() => {
    const fetchSummary = async content => {
      setIsConnectorMissing(false);
      setIsGenerating(true);
      setHasAlertSummary(true);
      const rawResponse = await sendMessage({
        ...content,
        promptIds: {
          promptGroupId: 'aiForSoc',
          promptId: 'alertSummarySystemPrompt'
        },
        query: {
          content_references_disabled: true
        }
      });
      let responseSummary;
      let responseRecommendedActions;
      try {
        const parsedResponse = JSON.parse(rawResponse.response);
        responseSummary = parsedResponse.summary;
        responseRecommendedActions = parsedResponse.recommendedActions;
      } catch (e) {
        // AI did not return the expected JSON
        responseSummary = rawResponse.response;
      }
      if (!rawResponse.isError) {
        if (fetchedAlertSummary.data.length > 0) {
          await bulkUpdate({
            alertSummary: {
              update: [{
                id: fetchedAlertSummary.data[0].id,
                summary: responseSummary,
                ...(responseRecommendedActions ? {
                  recommendedActions: responseRecommendedActions
                } : {}),
                replacements: content.replacements
              }]
            }
          });
        } else {
          await bulkUpdate({
            alertSummary: {
              create: [{
                alertId,
                summary: responseSummary,
                ...(responseRecommendedActions ? {
                  recommendedActions: responseRecommendedActions
                } : {}),
                replacements: content.replacements
              }]
            }
          });
        }
        await refetchAlertSummary();
      } else {
        if (responseSummary.includes('Failed to load action')) {
          setIsConnectorMissing(true);
        }
        setAlertSummary(showAnonymizedValues ? responseSummary : (0, _elasticAssistantCommon.replaceAnonymizedValuesWithOriginalValues)({
          messageContent: responseSummary,
          replacements: content.replacements
        }));
      }
      setIsGenerating(false);
    };
    if (messageAndReplacements !== null) fetchSummary(messageAndReplacements);
  }, [alertId, bulkUpdate, fetchedAlertSummary.data, messageAndReplacements, refetchAlertSummary, sendMessage, showAnonymizedValues]);
  (0, _react.useEffect)(() => {
    return () => {
      abortStream();
    };
  }, [abortStream]);
  return {
    alertSummary,
    hasAlertSummary,
    fetchAISummary,
    isConnectorMissing,
    isLoading: isGenerating,
    messageAndReplacements,
    recommendedActions
  };
};
exports.useAlertSummary = useAlertSummary;