"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useListExceptionItems = void 0;
var _react = require("react");
var _securitysolutionExceptionListComponents = require("@kbn/securitysolution-exception-list-components");
var _kibana = require("../../../common/lib/kibana");
var _api = require("../../api");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useListExceptionItems = ({
  list,
  deleteToastTitle,
  deleteToastBody,
  errorToastTitle,
  errorToastBody,
  onEditListExceptionItem,
  onFinishFetchingExceptions
}) => {
  const {
    services
  } = (0, _kibana.useKibana)();
  const {
    http
  } = services;
  const toasts = (0, _kibana.useToasts)();
  const [exceptions, setExceptions] = (0, _react.useState)([]);
  const [exceptionListReferences, setExceptionListReferences] = (0, _react.useState)({});
  const [pagination, setPagination] = (0, _react.useState)({
    pageIndex: 0,
    pageSize: 0,
    totalItemCount: 0
  });
  const [lastUpdated, setLastUpdated] = (0, _react.useState)(null);
  const [viewerStatus, setViewerStatus] = (0, _react.useState)('');
  const handleErrorStatus = (0, _react.useCallback)((error, errorTitle, errorDescription) => {
    toasts === null || toasts === void 0 ? void 0 : toasts.addError(error, {
      title: errorTitle || errorToastTitle,
      toastMessage: errorDescription || errorToastBody
    });
    setViewerStatus(_securitysolutionExceptionListComponents.ViewerStatus.ERROR);
  }, [errorToastBody, errorToastTitle, toasts]);
  const getReferences = (0, _react.useCallback)(async () => {
    try {
      const result = await (0, _api.getExceptionItemsReferences)([list]);
      setExceptionListReferences(result);
    } catch (error) {
      handleErrorStatus(error);
    }
  }, [handleErrorStatus, list, setExceptionListReferences]);
  const updateViewer = (0, _react.useCallback)((paginationResult, dataLength, viewStatus) => {
    setPagination(paginationResult);
    setLastUpdated(Date.now());
    setTimeout(() => {
      if (viewStatus === _securitysolutionExceptionListComponents.ViewerStatus.EMPTY_SEARCH) return setViewerStatus(!dataLength ? viewStatus : '');
      setViewerStatus(!dataLength ? _securitysolutionExceptionListComponents.ViewerStatus.EMPTY : '');
    }, 200);
  }, []);
  const fetchItems = (0, _react.useCallback)(async (options, viewStatus) => {
    try {
      setViewerStatus(_securitysolutionExceptionListComponents.ViewerStatus.LOADING);
      const {
        data,
        pagination: paginationResult
      } = await (0, _api.fetchListExceptionItems)({
        http,
        ...(0, _api.prepareFetchExceptionItemsParams)(null, list, options)
      });
      setExceptions(data);
      getReferences();
      updateViewer(paginationResult, data.length, viewStatus);
      if (typeof onFinishFetchingExceptions === 'function') onFinishFetchingExceptions();
    } catch (error) {
      handleErrorStatus(error);
    }
  }, [http, list, getReferences, updateViewer, onFinishFetchingExceptions, handleErrorStatus]);
  const onDeleteException = (0, _react.useCallback)(async ({
    id,
    name,
    namespaceType
  }) => {
    try {
      setViewerStatus(_securitysolutionExceptionListComponents.ViewerStatus.LOADING);
      await (0, _api.deleteException)({
        id,
        http,
        namespaceType
      });
      toasts === null || toasts === void 0 ? void 0 : toasts.addSuccess({
        title: deleteToastTitle,
        text: typeof deleteToastBody === 'function' ? deleteToastBody(name) : ''
      });
      fetchItems();
    } catch (error) {
      handleErrorStatus(error);
    }
  }, [http, toasts, deleteToastTitle, deleteToastBody, fetchItems, handleErrorStatus]);
  const onEditExceptionItem = exception => {
    if (typeof onEditListExceptionItem === 'function') onEditListExceptionItem(exception);
  };
  const onPaginationChange = (0, _react.useCallback)(async options => {
    fetchItems(options);
  }, [fetchItems]);
  return {
    exceptions,
    lastUpdated,
    pagination,
    exceptionViewerStatus: viewerStatus,
    ruleReferences: exceptionListReferences,
    fetchItems,
    onDeleteException,
    onEditExceptionItem,
    onPaginationChange
  };
};
exports.useListExceptionItems = useListExceptionItems;