"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchFilters = exports.getNormalizedSeverity = exports.getExportedRulesDetails = exports.getExportedRulesCounts = exports.caseInsensitiveSort = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const caseInsensitiveSort = tags => {
  return tags.sort((a, b) => a.toLowerCase().localeCompare(b.toLowerCase())); // Case insensitive
};
exports.caseInsensitiveSort = caseInsensitiveSort;
const getSearchFilters = ({
  query,
  searchValue,
  filterOptions,
  defaultSearchTerm
}) => {
  const fieldClauses = query === null || query === void 0 ? void 0 : query.ast.getFieldClauses();
  if (fieldClauses != null && fieldClauses.length > 0) {
    const filtersReduced = fieldClauses.reduce((acc, {
      field,
      value
    }) => {
      acc[field] = `${value}`;
      return acc;
    }, filterOptions);
    return filtersReduced;
  }
  return {
    [defaultSearchTerm]: searchValue
  };
};

/**
 * This function helps to parse NDJSON with exported rules
 * and retrieve the metadata of exported rules.
 *
 * @param blob a Blob received from an _export endpoint
 * @returns Export details
 */
exports.getSearchFilters = getSearchFilters;
const getExportedRulesDetails = async blob => {
  const blobContent = await blob.text();
  // The Blob content is an NDJSON file, the last line of which contains export details.
  const exportDetailsJson = blobContent.split('\n').filter(Boolean).slice(-1)[0];
  const exportDetails = JSON.parse(exportDetailsJson);
  return exportDetails;
};

/**
 * This function helps to parse NDJSON with exported rules
 * and retrieve the object with counts of successfully exported/missing/total rules.
 *
 * @param blob a Blob received from an _export endpoint
 * @returns Object of exported rules counts
 */
exports.getExportedRulesDetails = getExportedRulesDetails;
const getExportedRulesCounts = async blob => {
  const details = await getExportedRulesDetails(blob);
  return {
    skipped: 0,
    succeeded: details.exported_rules_count,
    failed: details.missing_rules_count,
    total: details.exported_rules_count + details.missing_rules_count
  };
};
exports.getExportedRulesCounts = getExportedRulesCounts;
const NormalizedSeverity = {
  low: 0,
  medium: 1,
  high: 2,
  critical: 3
};
const getNormalizedSeverity = severity => {
  var _NormalizedSeverity$s;
  return (_NormalizedSeverity$s = NormalizedSeverity[severity]) !== null && _NormalizedSeverity$s !== void 0 ? _NormalizedSeverity$s : -1;
};
exports.getNormalizedSeverity = getNormalizedSeverity;