"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSubfieldChanges = void 0;
var _utils = require("../utils");
var _data_source = require("./data_source");
var _kql_query = require("./kql_query");
var _eql_query = require("./eql_query");
var _esql_query = require("./esql_query");
var _threat_query = require("./threat_query");
var _rule_schedule = require("./rule_schedule");
var _rule_name_override = require("./rule_name_override");
var _timestamp_override = require("./timestamp_override");
var _timeline_template = require("./timeline_template");
var _building_block = require("./building_block");
var _threshold = require("./threshold");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Splits a field into subfields and returns the changes between the old and new subfield values.
 *
 * @param fieldName - The name of the field for which subfield changes are to be computed.
 * @param oldFieldValue - The old value of the field.
 * @param newFieldValue - The new value of the field.
 * @returns - An array of subfield changes.
 */
const getSubfieldChanges = (fieldName, oldFieldValue, newFieldValue) => {
  switch (fieldName) {
    /*
      Typecasting `oldFieldValue` and `newFieldValue` to corresponding field
      type `DiffableAllFields[*]` is required here since `oldFieldValue` and
      `newFieldValue` concrete types depend on `fieldName` but TS doesn't track that.
    */
    case 'data_source':
      return (0, _data_source.getSubfieldChangesForDataSource)(oldFieldValue, newFieldValue);
    case 'kql_query':
      return (0, _kql_query.getSubfieldChangesForKqlQuery)(oldFieldValue, newFieldValue);
    case 'eql_query':
      return (0, _eql_query.getSubfieldChangesForEqlQuery)(oldFieldValue, newFieldValue);
    case 'esql_query':
      return (0, _esql_query.getSubfieldChangesForEsqlQuery)(oldFieldValue, newFieldValue);
    case 'threat_query':
      return (0, _threat_query.getSubfieldChangesForThreatQuery)(oldFieldValue, newFieldValue);
    case 'rule_schedule':
      return (0, _rule_schedule.getSubfieldChangesForRuleSchedule)(oldFieldValue, newFieldValue);
    case 'rule_name_override':
      return (0, _rule_name_override.getSubfieldChangesForRuleNameOverride)(oldFieldValue, newFieldValue);
    case 'timestamp_override':
      return (0, _timestamp_override.getSubfieldChangesForTimestampOverride)(oldFieldValue, newFieldValue);
    case 'timeline_template':
      return (0, _timeline_template.getSubfieldChangesForTimelineTemplate)(oldFieldValue, newFieldValue);
    case 'building_block':
      return (0, _building_block.getSubfieldChangesForBuildingBlock)(oldFieldValue, newFieldValue);
    case 'threshold':
      return (0, _threshold.getSubfieldChangesForThreshold)(oldFieldValue, newFieldValue);
    default:
      const oldFieldValueStringified = (0, _utils.stringifyToSortedJson)(oldFieldValue);
      const newFieldValueStringified = (0, _utils.stringifyToSortedJson)(newFieldValue);
      if (oldFieldValueStringified === newFieldValueStringified) {
        return [];
      }
      return [{
        subfieldName: fieldName,
        oldSubfieldValue: oldFieldValueStringified,
        newSubfieldValue: newFieldValueStringified
      }];
  }
};
exports.getSubfieldChanges = getSubfieldChanges;