"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DiffableThresholdFields = exports.DiffableThreatMatchFields = exports.DiffableSavedQueryFields = exports.DiffableRuleTypes = exports.DiffableRule = exports.DiffableNewTermsFields = exports.DiffableMachineLearningFields = exports.DiffableFieldsByTypeUnion = exports.DiffableEsqlFields = exports.DiffableEqlFields = exports.DiffableCustomQueryFields = exports.DiffableCommonFields = exports.DiffableAllFields = exports.DIFFABLE_RULE_TYPE_FIELDS_MAP = void 0;
var _zod = require("@kbn/zod");
var _rule_schema = require("../../../../model/rule_schema");
var _diffable_field_types = require("./diffable_field_types");
var _rule_schedule = require("../../../../model/rule_schema/rule_schedule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DiffableCommonFields = exports.DiffableCommonFields = _zod.z.object({
  // Technical fields
  // NOTE: We might consider removing them from the schema and returning from the API
  // not via the fields diff, but via dedicated properties in the response body.
  rule_id: _rule_schema.RuleSignatureId,
  version: _rule_schema.RuleVersion,
  // Main domain fields
  name: _rule_schema.RuleName,
  tags: _rule_schema.RuleTagArray,
  description: _rule_schema.RuleDescription,
  severity: _rule_schema.Severity,
  severity_mapping: _rule_schema.SeverityMapping,
  risk_score: _rule_schema.RiskScore,
  risk_score_mapping: _rule_schema.RiskScoreMapping,
  // About -> Advanced settings
  references: _rule_schema.RuleReferenceArray,
  false_positives: _rule_schema.RuleFalsePositiveArray,
  threat: _rule_schema.ThreatArray,
  note: _rule_schema.InvestigationGuide,
  setup: _rule_schema.SetupGuide,
  related_integrations: _rule_schema.RelatedIntegrationArray,
  required_fields: _rule_schema.RequiredFieldArray,
  // Other domain fields
  rule_schedule: _rule_schedule.RuleSchedule,
  // NOTE: new field
  max_signals: _rule_schema.MaxSignals,
  // Optional fields
  investigation_fields: _rule_schema.InvestigationFields.optional(),
  rule_name_override: _diffable_field_types.RuleNameOverrideObject.optional(),
  // NOTE: new field
  timestamp_override: _diffable_field_types.TimestampOverrideObject.optional(),
  // NOTE: new field
  timeline_template: _diffable_field_types.TimelineTemplateReference.optional(),
  // NOTE: new field
  building_block: _diffable_field_types.BuildingBlockObject.optional() // NOTE: new field
});
const DiffableCustomQueryFields = exports.DiffableCustomQueryFields = _zod.z.object({
  type: _zod.z.literal('query'),
  kql_query: _diffable_field_types.RuleKqlQuery,
  // NOTE: new field
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableSavedQueryFields = exports.DiffableSavedQueryFields = _zod.z.object({
  type: _zod.z.literal('saved_query'),
  kql_query: _diffable_field_types.RuleKqlQuery,
  // NOTE: new field
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableEqlFields = exports.DiffableEqlFields = _zod.z.object({
  type: _zod.z.literal('eql'),
  eql_query: _diffable_field_types.RuleEqlQuery,
  // NOTE: new field
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  alert_suppression: _rule_schema.AlertSuppression.optional()
});

// this is a new type of rule, no prebuilt rules created yet.
// new properties might be added here during further rule type development

const DiffableEsqlFields = exports.DiffableEsqlFields = _zod.z.object({
  type: _zod.z.literal('esql'),
  esql_query: _diffable_field_types.RuleEsqlQuery,
  // NOTE: new field
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableThreatMatchFields = exports.DiffableThreatMatchFields = _zod.z.object({
  type: _zod.z.literal('threat_match'),
  kql_query: _diffable_field_types.RuleKqlQuery,
  // NOTE: new field
  threat_query: _diffable_field_types.InlineKqlQuery,
  // NOTE: new field
  threat_index: _rule_schema.ThreatIndex,
  threat_mapping: _rule_schema.ThreatMapping,
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  threat_indicator_path: _rule_schema.ThreatIndicatorPath.optional(),
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableThresholdFields = exports.DiffableThresholdFields = _zod.z.object({
  type: _zod.z.literal('threshold'),
  kql_query: _diffable_field_types.RuleKqlQuery,
  // NOTE: new field
  threshold: _rule_schema.Threshold,
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  alert_suppression: _rule_schema.ThresholdAlertSuppression.optional()
});
const DiffableMachineLearningFields = exports.DiffableMachineLearningFields = _zod.z.object({
  type: _zod.z.literal('machine_learning'),
  machine_learning_job_id: _rule_schema.MachineLearningJobId,
  anomaly_threshold: _rule_schema.AnomalyThreshold,
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableNewTermsFields = exports.DiffableNewTermsFields = _zod.z.object({
  type: _zod.z.literal('new_terms'),
  kql_query: _diffable_field_types.InlineKqlQuery,
  // NOTE: new field
  new_terms_fields: _rule_schema.NewTermsFields,
  history_window_start: _rule_schema.HistoryWindowStart,
  data_source: _diffable_field_types.RuleDataSource.optional(),
  // NOTE: new field
  alert_suppression: _rule_schema.AlertSuppression.optional()
});
const DiffableFieldsByTypeUnion = exports.DiffableFieldsByTypeUnion = _zod.z.discriminatedUnion('type', [DiffableCustomQueryFields, DiffableSavedQueryFields, DiffableEqlFields, DiffableEsqlFields, DiffableThreatMatchFields, DiffableThresholdFields, DiffableMachineLearningFields, DiffableNewTermsFields]);

/**
 * Represents a normalized rule object that is suitable for passing to the diff algorithm.
 * Every top-level field of a diffable rule can be compared separately on its own.
 *
 * It's important to do such normalization because:
 *
 * 1. We need to compare installed rules with prebuilt rule content. These objects have similar but not exactly
 * the same interfaces. In order to compare them we need to convert them to a common interface.
 *
 * 2. It only makes sense to compare certain rule fields in combination with other fields. For example,
 * we combine `index` and `data_view_id` fields into a `RuleDataSource` object, so that later we could
 * calculate a diff for this whole object. If we don't combine them the app would successfully merge the
 * following values independently from each other without a conflict:
 *
 *   Base version: index=[logs-*], data_view_id=undefined
 *   Current version: index=[], data_view_id=some-data-view // user switched to a data view
 *   Target version: index=[logs-*, filebeat-*], data_view_id=undefined // Elastic added a new index pattern
 *   Merged version: index=[filebeat-*], data_view_id=some-data-view ???
 *
 * Instead, semantically such change represents a conflict because the data source of the rule was changed
 * in a potentially incompatible way, and the user might want to review the change and resolve it manually.
 * The user must either pick index patterns or a data view, but not both at the same time.
 *
 * NOTE: Every top-level field in a DiffableRule MUST BE LOGICALLY INDEPENDENT from other
 * top-level fields.
 */

const DiffableRule = exports.DiffableRule = _zod.z.intersection(DiffableCommonFields, DiffableFieldsByTypeUnion);

/**
 * Union of all possible rule types
 */

const DiffableRuleTypes = exports.DiffableRuleTypes = _zod.z.union([DiffableCustomQueryFields.shape.type, DiffableSavedQueryFields.shape.type, DiffableEqlFields.shape.type, DiffableEsqlFields.shape.type, DiffableThreatMatchFields.shape.type, DiffableThresholdFields.shape.type, DiffableMachineLearningFields.shape.type, DiffableNewTermsFields.shape.type]);

/**
 * This is a merge of all fields from all rule types into a single TS type.
 * This is NOT a union discriminated by rule type, as DiffableRule is.
 */

const DiffableAllFields = exports.DiffableAllFields = DiffableCommonFields.merge(DiffableCustomQueryFields.omit({
  type: true
})).merge(DiffableSavedQueryFields.omit({
  type: true
})).merge(DiffableEqlFields.omit({
  type: true
})).merge(DiffableEsqlFields.omit({
  type: true
})).merge(DiffableThreatMatchFields.omit({
  type: true
})).merge(DiffableThresholdFields.omit({
  type: true
})).merge(DiffableMachineLearningFields.omit({
  type: true
})).merge(DiffableNewTermsFields.omit({
  type: true
})).merge(_zod.z.object({
  type: DiffableRuleTypes
}));
const getRuleTypeFields = schema => Object.keys(schema.shape);
const createDiffableFieldsPerRuleType = specificFields => [...getRuleTypeFields(DiffableCommonFields), ...getRuleTypeFields(specificFields)];
const DIFFABLE_RULE_TYPE_FIELDS_MAP = exports.DIFFABLE_RULE_TYPE_FIELDS_MAP = new Map([['query', createDiffableFieldsPerRuleType(DiffableCustomQueryFields)], ['saved_query', createDiffableFieldsPerRuleType(DiffableSavedQueryFields)], ['eql', createDiffableFieldsPerRuleType(DiffableEqlFields)], ['esql', createDiffableFieldsPerRuleType(DiffableEsqlFields)], ['threat_match', createDiffableFieldsPerRuleType(DiffableThreatMatchFields)], ['threshold', createDiffableFieldsPerRuleType(DiffableThresholdFields)], ['machine_learning', createDiffableFieldsPerRuleType(DiffableMachineLearningFields)], ['new_terms', createDiffableFieldsPerRuleType(DiffableNewTermsFields)]]);