"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.InvalidGrantError = void 0;
exports.getDetailedErrorMessage = getDetailedErrorMessage;
exports.getErrorStatusCode = getErrorStatusCode;
exports.isCredentialMismatchError = isCredentialMismatchError;
exports.isExpiredOrInvalidRefreshTokenError = isExpiredOrInvalidRefreshTokenError;
exports.wrapError = wrapError;
exports.wrapIntoCustomErrorResponse = wrapIntoCustomErrorResponse;
var _elasticsearch = require("@elastic/elasticsearch");
var _boom = _interopRequireDefault(require("@hapi/boom"));
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function wrapError(error) {
  return _boom.default.boomify(error, {
    statusCode: getErrorStatusCode(error)
  });
}

/**
 * Wraps error into error suitable for Core's custom error response.
 * @param error Any error instance.
 */
function wrapIntoCustomErrorResponse(error) {
  const wrappedError = wrapError(error);
  return {
    body: wrappedError,
    headers: wrappedError.output.headers,
    statusCode: wrappedError.output.statusCode
  };
}

/**
 * Extracts error code from Boom and Elasticsearch "native" errors.
 * @param error Error instance to extract status code from.
 */
function getErrorStatusCode(error) {
  if (error instanceof _elasticsearch.errors.ResponseError) {
    return error.statusCode;
  }
  return _boom.default.isBoom(error) ? error.output.statusCode : error.statusCode || error.status;
}

/**
 * Extracts detailed error message from Boom and Elasticsearch "native" errors. It's supposed to be
 * only logged on the server side and never returned to the client as it may contain sensitive
 * information.
 * @param error Error instance to extract message from.
 */
function getDetailedErrorMessage(error) {
  if (error instanceof _elasticsearch.errors.ResponseError) {
    return JSON.stringify(error.body);
  }
  if (_boom.default.isBoom(error)) {
    return JSON.stringify(error.output.payload);
  }
  return error.message;
}
function isExpiredOrInvalidRefreshTokenError(error) {
  var _error$body, _error$body$error_des, _error$body2, _error$body2$error_de;
  return ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : (_error$body$error_des = _error$body.error_description) === null || _error$body$error_des === void 0 ? void 0 : _error$body$error_des.includes('token has already been refreshed')) || ((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : (_error$body2$error_de = _error$body2.error_description) === null || _error$body2$error_de === void 0 ? void 0 : _error$body2$error_de.includes('could not refresh the requested token'));
}
function isCredentialMismatchError(error) {
  var _error$body3, _error$body3$error_de;
  return (_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : (_error$body3$error_de = _error$body3.error_description) === null || _error$body3$error_de === void 0 ? void 0 : _error$body3$error_de.includes('tokens must be refreshed by the creating client');
}
class InvalidGrantError extends Error {
  constructor(message) {
    super(message);
    this.name = 'InvalidGrantError';
    Object.setPrototypeOf(this, InvalidGrantError.prototype);
  }
  static expiredOrInvalidRefreshToken() {
    return new InvalidGrantError('Your session has expired because your refresh token is no longer valid. Please log in again.');
  }
  static credentialMismatch() {
    return new InvalidGrantError('Your session could not be refreshed due to a system misconfiguration. Please contact your administrator for assistance.');
  }
}
exports.InvalidGrantError = InvalidGrantError;