"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerInstallationRoutes = void 0;
var _coreSecurityServer = require("@kbn/core-security-server");
var _configSchema = require("@kbn/config-schema");
var _inferenceCommon = require("@kbn/inference-common");
var _installation = require("../../common/http_api/installation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerInstallationRoutes = ({
  router,
  getServices
}) => {
  router.get({
    path: _installation.INSTALLATION_STATUS_API_PATH,
    validate: {
      query: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        })
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$query;
    const {
      installClient,
      documentationManager
    } = getServices();
    const inferenceId = (_req$query = req.query) === null || _req$query === void 0 ? void 0 : _req$query.inferenceId;
    const installStatus = await installClient.getInstallationStatus({
      inferenceId
    });
    const {
      status: overallStatus
    } = await documentationManager.getStatus({
      inferenceId
    });
    return res.ok({
      body: {
        inferenceId,
        perProducts: installStatus,
        overall: overallStatus
      }
    });
  });
  router.post({
    path: _installation.INSTALL_ALL_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        })
      })
    },
    options: {
      access: 'internal',
      timeout: {
        idleSocket: 20 * 60 * 1000
      } // install can take time.
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$body;
    const {
      documentationManager
    } = getServices();
    const inferenceId = (_req$body = req.body) === null || _req$body === void 0 ? void 0 : _req$body.inferenceId;
    await documentationManager.install({
      request: req,
      force: false,
      wait: true,
      inferenceId
    });

    // check status after installation in case of failure
    const {
      status,
      installStatus
    } = await documentationManager.getStatus({
      inferenceId
    });
    let failureReason = null;
    if (status === 'error' && installStatus) {
      failureReason = Object.values(installStatus).filter(product => product.status === 'error' && product.failureReason).map(product => product.failureReason).join('\n');
    }
    return res.ok({
      body: {
        installed: status === 'installed',
        ...(failureReason ? {
          failureReason
        } : {})
      }
    });
  });
  router.post({
    path: _installation.UNINSTALL_ALL_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        inferenceId: _configSchema.schema.string({
          defaultValue: _inferenceCommon.defaultInferenceEndpoints.ELSER
        })
      })
    },
    options: {
      access: 'internal'
    },
    security: {
      authz: {
        requiredPrivileges: [_coreSecurityServer.ApiPrivileges.manage('llm_product_doc')]
      }
    }
  }, async (ctx, req, res) => {
    var _req$body2;
    const {
      documentationManager
    } = getServices();
    await documentationManager.uninstall({
      request: req,
      wait: true,
      inferenceId: (_req$body2 = req.body) === null || _req$body2 === void 0 ? void 0 : _req$body2.inferenceId
    });
    return res.ok({
      body: {
        success: true
      }
    });
  });
};
exports.registerInstallationRoutes = registerInstallationRoutes;