"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerExpressionsLanguage = registerExpressionsLanguage;
var _monaco = require("@kbn/monaco");
var _common = require("../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Extends the default type for a Monarch language so we can use
 * attribute references (like @keywords to reference the keywords list)
 * in the defined tokenizer
 */

/**
 * Defines the Monarch tokenizer for syntax highlighting in Monaco of the
 * expression language. The tokenizer defines a set of regexes and actions/tokens
 * to mark the detected words/characters.
 *
 * For more information, the Monarch documentation can be found here:
 * https://microsoft.github.io/monaco-editor/monarch.html
 */
const expressionsLanguage = {
  keywords: [],
  deprecated: [],
  symbols: /[=|]/,
  escapes: /\\(?:[abfnrtv\\"']|x[0-9A-Fa-f]{1,4}|u[0-9A-Fa-f]{4}|U[0-9A-Fa-f]{8})/,
  digits: /\d+(_+\d+)*/,
  boolean: ['true', 'false'],
  null: ['null'],
  tokenizer: {
    root: [[/[{}]/, 'delimiter.bracket'], {
      include: 'common'
    }],
    common: [
    // arguments (some args share the same name as functions which are keywords)
    [/[a-z_$][\w$]*=/, 'identifier'],
    // identifiers and keywords
    [/[a-z_$][\w$]*/, {
      cases: {
        '@keywords': {
          cases: {
            '@deprecated': 'keyword.deprecated',
            '@default': 'keyword'
          }
        },
        '@null': 'keyword',
        '@boolean': 'keyword',
        '@default': 'identifier'
      }
    }], [/(@digits)/, 'number'], [/"/, 'string', '@string_double'], [/'/, 'string', '@string_single'], [/@symbols/, 'delimiter'], [/\/\*/, 'comment', '@multiline_comment'], [/\/\/.*$/, 'comment']],
    string_double: [[/[^\\"]+/, 'string'], [/@escapes/, 'string.escape'], [/\\./, 'string.escape.invalid'], [/"/, 'string', '@pop']],
    string_single: [[/[^\\']+/, 'string'], [/@escapes/, 'string.escape'], [/\\./, 'string.escape.invalid'], [/'/, 'string', '@pop']],
    bracketCounting: [[/\{/, 'delimiter.bracket', '@bracketCounting'], [/\}/, 'delimiter.bracket', '@pop'], {
      include: 'common'
    }],
    multiline_comment: [[/[^\/*]+/, 'comment'], ['\\*/', 'comment', '@pop'], [/[\/*]/, 'comment']]
  }
};
function registerExpressionsLanguage(functions) {
  expressionsLanguage.keywords = functions.map(fn => fn.name);
  expressionsLanguage.deprecated = functions.filter(fn => fn.deprecated).map(fn => fn.name);
  _monaco.monaco.languages.onLanguage(_common.EXPRESSIONS_LANGUAGE_ID, () => {
    _monaco.monaco.languages.setMonarchTokensProvider(_common.EXPRESSIONS_LANGUAGE_ID, expressionsLanguage);
  });
  _monaco.monaco.languages.register({
    id: _common.EXPRESSIONS_LANGUAGE_ID
  });
}