"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.joinByKey = joinByKey;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Joins a list of records by a given key. Key can be any type of value, from
 * strings to plain objects, as long as it is present in all records. `isEqual`
 * is used for comparing keys.
 *
 * UnionToIntersection is needed to get all keys of union types, see below for
 * example.
 *
 const agentNames = [{ serviceName: '', agentName: '' }];
 const transactionRates = [{ serviceName: '', transactionsPerMinute: 1 }];
 const flattened = joinByKey(
  [...agentNames, ...transactionRates],
  'serviceName'
 );
*/

function joinByKey(items, key, mergeFn = (a, b) => (0, _lodash.merge)({}, a, b)) {
  const keys = (0, _lodash.castArray)(key);
  return items.reduce((prev, current) => {
    let item = prev.find(prevItem => keys.every(k => (0, _lodash.isEqual)(prevItem[k], current[k])));
    if (!item) {
      item = {
        ...current
      };
      prev.push(item);
    } else {
      (0, _lodash.pull)(prev, item).push(mergeFn(item, current));
    }
    return prev;
  }, []);
}