"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isSemanticTextUnsupportedError = isSemanticTextUnsupportedError;
exports.runStartupMigrations = runStartupMigrations;
var _elasticsearch = require("@elastic/elasticsearch");
var _lockManager = require("@kbn/lock-manager");
var _ = require("..");
var _reindex_knowledge_base = require("../knowledge_base_service/reindex_knowledge_base");
var _populate_missing_semantic_text_fields = require("./populate_missing_semantic_text_fields");
var _has_kb_index = require("../knowledge_base_service/has_kb_index");
var _update_existing_index_assets = require("../index_assets/update_existing_index_assets");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const PLUGIN_STARTUP_LOCK_ID = 'observability_ai_assistant:startup_migrations';

// This function performs necessary startup migrations for the observability AI assistant:
// 1. Updates index assets to ensure mappings are correct
// 2. If the knowledge base index does not support the `semantic_text` field, it is re-indexed.
// 3. Populates the `semantic_text` field for knowledge base entries
async function runStartupMigrations({
  core,
  logger,
  config
}) {
  // update index assets to ensure mappings are correct
  await (0, _update_existing_index_assets.updateExistingIndexAssets)({
    logger,
    core
  });
  const [coreStart] = await core.getStartServices();
  const esClient = coreStart.elasticsearch.client;
  const lmService = new _lockManager.LockManagerService(core, logger);
  await lmService.withLock(PLUGIN_STARTUP_LOCK_ID, async () => {
    const doesKbIndexExist = await (0, _has_kb_index.hasKbWriteIndex)({
      esClient
    });
    if (!doesKbIndexExist) {
      logger.info('Knowledge base index does not exist. Aborting updating index assets');
      return;
    }
    const isKbSemanticTextCompatible = await isKnowledgeBaseSemanticTextCompatible({
      logger,
      esClient
    });
    if (!isKbSemanticTextCompatible) {
      await (0, _reindex_knowledge_base.reIndexKnowledgeBaseWithLock)({
        core,
        logger,
        esClient
      });
    }
    await (0, _populate_missing_semantic_text_fields.populateMissingSemanticTextFieldWithLock)({
      core,
      logger,
      config,
      esClient
    });
  }).catch(error => {
    // we should propogate the error if it is not a LockAcquisitionError
    if (!(0, _lockManager.isLockAcquisitionError)(error)) {
      throw error;
    }
    logger.info('Startup migrations are already in progress. Aborting startup migrations');
  });
}

// Checks if the knowledge base index supports `semantic_text`
// If the index was created before version 8.11, it requires re-indexing to support the `semantic_text` field.
async function isKnowledgeBaseSemanticTextCompatible({
  logger,
  esClient
}) {
  var _settings$index$versi, _settings$index, _settings$index$versi2;
  const indexSettingsResponse = await esClient.asInternalUser.indices.getSettings({
    index: _.resourceNames.writeIndexAlias.kb
  });
  const results = Object.entries(indexSettingsResponse);
  if (results.length === 0) {
    logger.debug('No knowledge base indices found. Skipping re-indexing.');
    return true;
  }
  const [indexName, {
    settings
  }] = results[0];
  const createdVersion = parseInt((_settings$index$versi = settings === null || settings === void 0 ? void 0 : (_settings$index = settings.index) === null || _settings$index === void 0 ? void 0 : (_settings$index$versi2 = _settings$index.version) === null || _settings$index$versi2 === void 0 ? void 0 : _settings$index$versi2.created) !== null && _settings$index$versi !== void 0 ? _settings$index$versi : '', 10);

  // Check if the index was created before version 8.11
  const versionThreshold = 8110000; // Version 8.11.0
  if (createdVersion >= versionThreshold) {
    logger.debug(`Knowledge base index "${indexName}" was created in version ${createdVersion}, and does not require re-indexing. Semantic text field is already supported. Aborting`);
    return true;
  }
  logger.info(`Knowledge base index was created in ${createdVersion} and must be re-indexed in order to support semantic_text field. Re-indexing now...`);
  return false;
}
function isSemanticTextUnsupportedError(error) {
  var _error$meta, _error$meta$body, _error$meta$body$erro, _error$meta$body$erro2;
  const semanticTextUnsupportedError = 'The [sparse_vector] field type is not supported on indices created on versions 8.0 to 8.10';
  const isSemanticTextUnspported = error instanceof _elasticsearch.errors.ResponseError && (error.message.includes(semanticTextUnsupportedError) || (// @ts-expect-error
  (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : (_error$meta$body$erro2 = _error$meta$body$erro.caused_by) === null || _error$meta$body$erro2 === void 0 ? void 0 : _error$meta$body$erro2.reason.includes(semanticTextUnsupportedError)));
  return isSemanticTextUnspported;
}