"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KnowledgeBaseService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = require("@hapi/boom");
var _lodash = require("lodash");
var _gptTokenizer = require("gpt-tokenizer");
var _lockManager = require("@kbn/lock-manager");
var _ = require("..");
var _types = require("../../../common/types");
var _get_access_query = require("../util/get_access_query");
var _get_category_query = require("../util/get_category_query");
var _get_space_query = require("../util/get_space_query");
var _inference_endpoint = require("../inference_endpoint");
var _recall_from_search_connectors = require("./recall_from_search_connectors");
var _has_kb_index = require("./has_kb_index");
var _reindex_knowledge_base = require("./reindex_knowledge_base");
var _run_startup_migrations = require("../startup_migrations/run_startup_migrations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function throwKnowledgeBaseNotReady(error) {
  throw (0, _boom.serverUnavailable)(`Knowledge base is not ready yet: ${error.message}`);
}
class KnowledgeBaseService {
  constructor(dependencies) {
    (0, _defineProperty2.default)(this, "recall", async ({
      user,
      queries,
      categories,
      namespace,
      esClient,
      uiSettingsClient,
      limit = {}
    }) => {
      var _limit$size, _limit$tokens;
      if (!this.dependencies.config.enableKnowledgeBase) {
        return [];
      }
      this.dependencies.logger.debug(() => `Recalling entries from KB for queries: "${JSON.stringify(queries)}"`);
      const [documentsFromKb, documentsFromConnectors] = await Promise.all([this.recallFromKnowledgeBase({
        user,
        queries,
        categories,
        namespace
      }).catch(error => {
        if ((0, _inference_endpoint.isInferenceEndpointMissingOrUnavailable)(error)) {
          throwKnowledgeBaseNotReady(error);
        }
        throw error;
      }), (0, _recall_from_search_connectors.recallFromSearchConnectors)({
        esClient,
        uiSettingsClient,
        queries,
        core: this.dependencies.core,
        logger: this.dependencies.logger
      }).catch(error => {
        this.dependencies.logger.debug('Error getting data from search indices');
        this.dependencies.logger.debug(error);
        return [];
      })]);
      this.dependencies.logger.debug(`documentsFromKb: ${JSON.stringify(documentsFromKb.slice(0, 5), null, 2)}`);
      this.dependencies.logger.debug(`documentsFromConnectors: ${JSON.stringify(documentsFromConnectors.slice(0, 5), null, 2)}`);
      const sortedEntries = (0, _lodash.orderBy)(documentsFromKb.concat(documentsFromConnectors), 'esScore', 'desc').slice(0, (_limit$size = limit.size) !== null && _limit$size !== void 0 ? _limit$size : 20);
      const maxTokens = (_limit$tokens = limit.tokens) !== null && _limit$tokens !== void 0 ? _limit$tokens : 4_000;
      let tokenCount = 0;
      const returnedEntries = [];
      for (const entry of sortedEntries) {
        returnedEntries.push(entry);
        tokenCount += (0, _gptTokenizer.encode)(entry.text).length;
        if (tokenCount >= maxTokens) {
          break;
        }
      }
      const droppedEntries = sortedEntries.length - returnedEntries.length;
      if (droppedEntries > 0) {
        this.dependencies.logger.info(`Dropped ${droppedEntries} entries because of token limit`);
      }
      return returnedEntries;
    });
    (0, _defineProperty2.default)(this, "getUserInstructions", async (namespace, user) => {
      if (!this.dependencies.config.enableKnowledgeBase) {
        return [];
      }
      const doesKbIndexExist = await (0, _has_kb_index.hasKbWriteIndex)({
        esClient: this.dependencies.esClient
      });
      if (!doesKbIndexExist) {
        return [];
      }
      try {
        const response = await this.dependencies.esClient.asInternalUser.search({
          index: _.resourceNames.writeIndexAlias.kb,
          query: {
            bool: {
              filter: [{
                term: {
                  type: _types.KnowledgeBaseType.UserInstruction
                }
              }, ...(0, _get_access_query.getAccessQuery)({
                user,
                namespace
              })]
            }
          },
          size: 500,
          _source: ['id', 'text', 'public']
        });
        return response.hits.hits.map(hit => {
          var _hit$_source$text, _hit$_source, _hit$_source2;
          return {
            id: hit._id,
            text: (_hit$_source$text = (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : _hit$_source.text) !== null && _hit$_source$text !== void 0 ? _hit$_source$text : '',
            public: (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : _hit$_source2.public
          };
        });
      } catch (error) {
        this.dependencies.logger.error('Failed to load instructions from knowledge base');
        this.dependencies.logger.error(error);
        return [];
      }
    });
    (0, _defineProperty2.default)(this, "getEntries", async ({
      query,
      sortBy,
      sortDirection,
      namespace
    }) => {
      if (!this.dependencies.config.enableKnowledgeBase) {
        return {
          entries: []
        };
      }
      try {
        const response = await this.dependencies.esClient.asInternalUser.search({
          index: _.resourceNames.writeIndexAlias.kb,
          query: {
            bool: {
              filter: [
              // filter by search query
              ...(query ? [{
                query_string: {
                  query: `${query}*`,
                  fields: ['doc_id', 'title']
                }
              }] : []), {
                // exclude user instructions
                bool: {
                  must_not: {
                    term: {
                      type: _types.KnowledgeBaseType.UserInstruction
                    }
                  }
                }
              },
              // filter by space
              ...(0, _get_space_query.getSpaceQuery)({
                namespace
              })]
            }
          },
          sort: sortBy === 'title' ? [{
            ['title.keyword']: {
              order: sortDirection
            }
          }] : [{
            [String(sortBy)]: {
              order: sortDirection
            }
          }],
          size: 500,
          _source: {
            excludes: ['confidence', 'is_correction'] // fields deprecated in https://github.com/elastic/kibana/pull/222814
          }
        });
        return {
          entries: response.hits.hits.map(hit => {
            var _title, _role;
            return {
              ...hit._source,
              title: (_title = hit._source.title) !== null && _title !== void 0 ? _title : hit._source.doc_id,
              // use `doc_id` as fallback title for backwards compatibility
              role: (_role = hit._source.role) !== null && _role !== void 0 ? _role : _types.KnowledgeBaseEntryRole.UserEntry,
              score: hit._score,
              id: hit._id
            };
          })
        };
      } catch (error) {
        if ((0, _inference_endpoint.isInferenceEndpointMissingOrUnavailable)(error)) {
          throwKnowledgeBaseNotReady(error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "hasEntries", async () => {
      var _response$hits$total$, _response$hits$total;
      const response = await this.dependencies.esClient.asInternalUser.search({
        index: _.resourceNames.writeIndexAlias.kb,
        size: 0,
        track_total_hits: 1,
        terminate_after: 1
      });
      const hitCount = typeof response.hits.total === 'number' ? response.hits.total : (_response$hits$total$ = (_response$hits$total = response.hits.total) === null || _response$hits$total === void 0 ? void 0 : _response$hits$total.value) !== null && _response$hits$total$ !== void 0 ? _response$hits$total$ : 0;
      return hitCount > 0;
    });
    (0, _defineProperty2.default)(this, "getPersonalUserInstructionId", async ({
      isPublic,
      user,
      namespace
    }) => {
      var _res$hits$hits$;
      if (!this.dependencies.config.enableKnowledgeBase) {
        return null;
      }
      const res = await this.dependencies.esClient.asInternalUser.search({
        index: _.resourceNames.writeIndexAlias.kb,
        query: {
          bool: {
            filter: [{
              term: {
                type: _types.KnowledgeBaseType.UserInstruction
              }
            }, {
              term: {
                public: isPublic
              }
            }, {
              term: {
                namespace
              }
            }, {
              bool: {
                should: [...(0, _get_access_query.getUserAccessFilters)(user)],
                minimum_should_match: 1
              }
            }]
          }
        },
        size: 1,
        _source: false
      });
      return (_res$hits$hits$ = res.hits.hits[0]) === null || _res$hits$hits$ === void 0 ? void 0 : _res$hits$hits$._id;
    });
    (0, _defineProperty2.default)(this, "getUuidFromDocId", async ({
      docId,
      user,
      namespace
    }) => {
      var _response$hits$hits$;
      const query = {
        bool: {
          filter: [{
            term: {
              doc_id: docId
            }
          },
          // exclude user instructions
          {
            bool: {
              must_not: {
                term: {
                  type: _types.KnowledgeBaseType.UserInstruction
                }
              }
            }
          },
          // restrict access to user's own entries
          ...(0, _get_access_query.getAccessQuery)({
            user,
            namespace
          })]
        }
      };
      const response = await this.dependencies.esClient.asInternalUser.search({
        size: 1,
        index: _.resourceNames.writeIndexAlias.kb,
        query,
        _source: false
      });
      return (_response$hits$hits$ = response.hits.hits[0]) === null || _response$hits$hits$ === void 0 ? void 0 : _response$hits$hits$._id;
    });
    (0, _defineProperty2.default)(this, "addEntry", async ({
      entry: {
        id,
        ...doc
      },
      user,
      namespace
    }) => {
      if (!this.dependencies.config.enableKnowledgeBase) {
        return;
      }
      try {
        const indexResult = await this.dependencies.esClient.asInternalUser.index({
          index: _.resourceNames.writeIndexAlias.kb,
          id,
          document: {
            '@timestamp': new Date().toISOString(),
            ...doc,
            ...(doc.text ? {
              semantic_text: doc.text
            } : {}),
            user,
            namespace
          },
          refresh: 'wait_for'
        });
        this.dependencies.logger.debug(`Entry added to knowledge base. title = "${doc.title}", user = "${user === null || user === void 0 ? void 0 : user.name}, namespace = "${namespace}", index = ${indexResult._index}, id = ${indexResult._id}`);
      } catch (error) {
        this.dependencies.logger.debug(`Failed to add entry to knowledge base ${error}`);
        if ((0, _inference_endpoint.isInferenceEndpointMissingOrUnavailable)(error)) {
          throwKnowledgeBaseNotReady(error);
        }
        if ((0, _run_startup_migrations.isSemanticTextUnsupportedError)(error)) {
          (0, _reindex_knowledge_base.reIndexKnowledgeBaseWithLock)({
            core: this.dependencies.core,
            logger: this.dependencies.logger,
            esClient: this.dependencies.esClient
          }).catch(e => {
            if ((0, _lockManager.isLockAcquisitionError)(e)) {
              this.dependencies.logger.info(`Re-indexing operation is already in progress`);
              return;
            }
            this.dependencies.logger.error(`Failed to re-index knowledge base: ${e.message}`);
          });
          throw (0, _boom.serverUnavailable)(`The index "${_.resourceNames.writeIndexAlias.kb}" does not support semantic text and must be reindexed. This re-index operation has been scheduled and will be started automatically. Please try again later.`);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "addBulkEntries", async ({
      entries,
      user,
      namespace
    }) => {
      if (!this.dependencies.config.enableKnowledgeBase) {
        return;
      }
      try {
        const result = await this.dependencies.esClient.asInternalUser.helpers.bulk({
          onDocument(doc) {
            return [{
              index: {
                _index: _.resourceNames.writeIndexAlias.kb,
                _id: doc.id
              }
            }, {
              '@timestamp': new Date().toISOString(),
              ...doc,
              ...(doc.text ? {
                semantic_text: doc.text
              } : {}),
              user,
              namespace
            }];
          },
          onDrop: doc => {
            var _doc$error;
            this.dependencies.logger.error(`Failed ingesting document: ${((_doc$error = doc.error) === null || _doc$error === void 0 ? void 0 : _doc$error.reason) || 'unknown reason'}`);
          },
          datasource: entries,
          refresh: 'wait_for',
          concurrency: 3,
          flushBytes: 100 * 1024,
          flushInterval: 1000,
          retries: 5
        });
        if (result.failed > 0) {
          throw Error(`Failed ingesting ${result.failed} documents.`);
        }
        this.dependencies.logger.debug(`Successfully added ${result.successful} entries to the knowledge base`);
      } catch (error) {
        this.dependencies.logger.error(`Failed to add entries to the knowledge base: ${error}`);
        if ((0, _inference_endpoint.isInferenceEndpointMissingOrUnavailable)(error)) {
          throwKnowledgeBaseNotReady(error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "deleteEntry", async ({
      id
    }) => {
      try {
        await this.dependencies.esClient.asInternalUser.delete({
          index: _.resourceNames.writeIndexAlias.kb,
          id,
          refresh: 'wait_for'
        });
        return Promise.resolve();
      } catch (error) {
        if ((0, _inference_endpoint.isInferenceEndpointMissingOrUnavailable)(error)) {
          throwKnowledgeBaseNotReady(error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getModelStatus", async () => {
      return (0, _inference_endpoint.getKbModelStatus)({
        core: this.dependencies.core,
        esClient: this.dependencies.esClient,
        logger: this.dependencies.logger,
        config: this.dependencies.config
      });
    });
    (0, _defineProperty2.default)(this, "getInferenceEndpointsForEmbedding", async () => {
      const {
        inferenceEndpoints
      } = await (0, _inference_endpoint.getInferenceEndpointsForEmbedding)({
        esClient: this.dependencies.esClient,
        logger: this.dependencies.logger
      });
      return inferenceEndpoints;
    });
    this.dependencies = dependencies;
  }
  async recallFromKnowledgeBase({
    queries,
    categories,
    namespace,
    user
  }) {
    const response = await this.dependencies.esClient.asInternalUser.search({
      index: [_.resourceNames.writeIndexAlias.kb],
      query: {
        bool: {
          should: queries.map(({
            text,
            boost = 1
          }) => ({
            semantic: {
              field: 'semantic_text',
              query: text,
              boost
            }
          })),
          filter: [...(0, _get_access_query.getAccessQuery)({
            user,
            namespace
          }), ...(0, _get_category_query.getCategoryQuery)({
            categories
          }),
          // exclude user instructions
          {
            bool: {
              must_not: {
                term: {
                  type: _types.KnowledgeBaseType.UserInstruction
                }
              }
            }
          }]
        }
      },
      size: 20,
      _source: {
        includes: ['text', 'labels', 'doc_id', 'title']
      }
    });
    return response.hits.hits.map(hit => {
      var _hit$_source3, _hit$_source4, _hit$_source$title, _hit$_source5, _hit$_source6;
      return {
        text: (_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : _hit$_source3.text,
        labels: (_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : _hit$_source4.labels,
        title: (_hit$_source$title = (_hit$_source5 = hit._source) === null || _hit$_source5 === void 0 ? void 0 : _hit$_source5.title) !== null && _hit$_source$title !== void 0 ? _hit$_source$title : (_hit$_source6 = hit._source) === null || _hit$_source6 === void 0 ? void 0 : _hit$_source6.doc_id,
        // use `doc_id` as fallback title for backwards compatibility
        esScore: hit._score,
        id: hit._id
      };
    });
  }
}
exports.KnowledgeBaseService = KnowledgeBaseService;