"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.catchFunctionNotFoundError = catchFunctionNotFoundError;
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _common = require("../../../../common");
var _conversation_complete = require("../../../../common/conversation_complete");
var _emit_with_concatenated_message = require("../../../../common/utils/emit_with_concatenated_message");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function appendFunctionLimitExceededErrorMessageToAssistantResponse() {
  return source$ => {
    return source$.pipe((0, _rxjs.filter)(msg => msg.type === _conversation_complete.StreamingChatResponseEventType.ChatCompletionChunk), (0, _emit_with_concatenated_message.emitWithConcatenatedMessage)(async concatenatedMessage => {
      return {
        ...concatenatedMessage,
        message: {
          ...concatenatedMessage.message,
          content: `${concatenatedMessage.message.content}\n\n${_i18n.i18n.translate('xpack.observabilityAiAssistant.functionCallLimitExceeded', {
            defaultMessage: '\n\nNote: the Assistant tried to call a function, even though the limit was exceeded'
          })}`,
          // remove any function call from the response so the stream can close
          function_call: {
            name: '',
            arguments: '',
            trigger: _common.MessageRole.Assistant
          }
        }
      };
    }));
  };
}

// we catch a function not found error, if:
// - the function limit has been exceeded,
// we append to the message to prevent the
// error going back to the LLM
// else: we complete the observable, and
// allow the LLM to correct the error
function catchFunctionNotFoundError(functionLimitExceeded) {
  return source$ => {
    const shared$ = source$.pipe((0, _rxjs.share)());
    const chunksWithoutErrors$ = shared$.pipe((0, _rxjs.filter)(event => event.type === _conversation_complete.StreamingChatResponseEventType.ChatCompletionChunk), (0, _rxjs.catchError)(() => (0, _rxjs.of)()));
    return shared$.pipe((0, _rxjs.catchError)(error => {
      if ((0, _conversation_complete.isFunctionNotFoundError)(error)) {
        if (functionLimitExceeded) {
          return chunksWithoutErrors$.pipe(appendFunctionLimitExceededErrorMessageToAssistantResponse());
        }
        return chunksWithoutErrors$.pipe((0, _emit_with_concatenated_message.emitWithConcatenatedMessage)());
      }
      return (0, _rxjs.throwError)(() => error);
    }));
  };
}