"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LogstashAgentMonitoring = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _create_query = require("./create_query");
var _get_high_level_stats = require("./get_high_level_stats");
var _constants = require("../../common/constants");
var _logstash_monitoring = require("./logstash_monitoring");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class LogstashAgentMonitoring {
  /*
   * Call the function for fetching and summarizing Logstash metrics for agent (LS integration) monitoring
   * @param {Object} callCluster - ES client
   * @param {Array} clusterUuids - List cluster UUIDs to retrieve metrics
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   * @return {Object} - Logstash stats in an object keyed by the cluster UUIDs
   * Note that, we try to fetch all metrics for the given time regardless of the cluster UUID
   * If metrics do not have UUID, metrics will be included in the monitoring cluster UUID
   */
  async collectMetrics(callCluster, clusterUuids, monitoringClusterUuid, start, end, options) {
    await this.fetchLogstashStats(callCluster, monitoringClusterUuid, start, end, options);
    const allHostIds = Object.values(options.allHostIds).flat();
    if (allHostIds.length > 0) {
      await this.fetchLogstashState(callCluster, allHostIds, monitoringClusterUuid, start, end, options);
    }
    return options.clusters;
  }
  setIndexPattern(monitoringType) {}

  /*
   * Update a clusters object with processed Logstash stats for agent monitoring
   * @param {Array} results - array of LogstashStats docs from ES
   * @param {Object} clusters - LogstashBaseStats in an object keyed by the cluster UUIDs
   * @param {Object} allEphemeralIds - EphemeralIds in an object keyed by cluster UUIDs to track the pipelines for the cluster
   * @param {Object} versions - Versions in an object keyed by cluster UUIDs to track the logstash versions for the cluster
   * @param {Object} plugins - plugin information keyed by cluster UUIDs to count the unique plugins
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   */
  processStatsResults(results, {
    clusters,
    allEphemeralIds,
    allHostIds,
    versions,
    plugins
  }, monitoringClusterUuid) {
    var _results$hits;
    const currHits = (results === null || results === void 0 ? void 0 : (_results$hits = results.hits) === null || _results$hits === void 0 ? void 0 : _results$hits.hits) || [];
    currHits.forEach(hit => {
      var _logstash, _logstash$elasticsear, _logstash$elasticsear2, _hit$_source, _hit$_source$logstash, _hit$_source$logstash2;
      // if orphan (no uuid) cluster found, report it with monitoring cluster UUID
      const clusterId = ((_logstash = hit._source.logstash) === null || _logstash === void 0 ? void 0 : (_logstash$elasticsear = _logstash.elasticsearch) === null || _logstash$elasticsear === void 0 ? void 0 : (_logstash$elasticsear2 = _logstash$elasticsear.cluster) === null || _logstash$elasticsear2 === void 0 ? void 0 : _logstash$elasticsear2.id) || [];
      const clusterUuid = clusterId[0] || monitoringClusterUuid;
      if (clusterUuid !== undefined && clusters[clusterUuid] === undefined) {
        clusters[clusterUuid] = (0, _logstash_monitoring.getLogstashBaseStats)();
        versions[clusterUuid] = new Map();
        plugins[clusterUuid] = new Map();
      }
      const logstashStats = (_hit$_source = hit._source) === null || _hit$_source === void 0 ? void 0 : (_hit$_source$logstash = _hit$_source.logstash) === null || _hit$_source$logstash === void 0 ? void 0 : (_hit$_source$logstash2 = _hit$_source$logstash.node) === null || _hit$_source$logstash2 === void 0 ? void 0 : _hit$_source$logstash2.stats;
      if (clusterUuid !== undefined && logstashStats !== undefined) {
        var _logstashStats$logsta, _logstashStats$logsta2, _hit$_source2, _hit$_source2$host, _logstashStats$logsta3;
        const clusterStats = clusters[clusterUuid].cluster_stats || {};
        clusterStats.monitoringClusterUuid = monitoringClusterUuid;
        clusters[clusterUuid].count = (clusters[clusterUuid].count || 0) + 1;
        const thisVersion = (_logstashStats$logsta = logstashStats.logstash) === null || _logstashStats$logsta === void 0 ? void 0 : _logstashStats$logsta.version;
        const a = versions[clusterUuid];
        (0, _get_high_level_stats.incrementByKey)(a, thisVersion);
        clusters[clusterUuid].versions = (0, _get_high_level_stats.mapToList)(a, 'version');
        const ephemeralId = (_logstashStats$logsta2 = logstashStats.logstash) === null || _logstashStats$logsta2 === void 0 ? void 0 : _logstashStats$logsta2.ephemeral_id;
        if (ephemeralId !== undefined) {
          allEphemeralIds[clusterUuid] = allEphemeralIds[clusterUuid] || [];
          if (!allEphemeralIds[clusterUuid].includes(ephemeralId)) {
            allEphemeralIds[clusterUuid].push(ephemeralId);
          }
        }
        const hostId = (_hit$_source2 = hit._source) === null || _hit$_source2 === void 0 ? void 0 : (_hit$_source2$host = _hit$_source2.host) === null || _hit$_source2$host === void 0 ? void 0 : _hit$_source2$host.id;
        if (hostId !== undefined) {
          allHostIds[clusterUuid] = allHostIds[clusterUuid] || [];
          if (!allHostIds[clusterUuid].includes(hostId)) {
            allHostIds[clusterUuid].push(hostId);
          }
        }
        const thisCollectionType = 'agent';
        if (!Object.hasOwn(clusterStats, 'collection_types')) {
          clusterStats.collection_types = {};
        }
        clusterStats.collection_types[thisCollectionType] = (clusterStats.collection_types[thisCollectionType] || 0) + 1;
        const pipelines = (logstashStats === null || logstashStats === void 0 ? void 0 : (_logstashStats$logsta3 = logstashStats.logstash) === null || _logstashStats$logsta3 === void 0 ? void 0 : _logstashStats$logsta3.pipelines) || [];
        if (!Object.hasOwn(clusterStats, 'pipelines')) {
          clusterStats.pipelines = {};
        }
        clusterStats.pipelines.count = pipelines.length;
        // TODO: add queue types of the pipelines with next iterations
      }
    });
  }

  /*
   * Update a clusters object with logstash state details for agent monitoring
   * @param {Array} results - array of LogstashState docs from ES
   * @param {Object} clusters - LogstashBaseStats in an object keyed by the cluster UUIDs
   * @param {Object} allEphemeralIds - EphemeralIds in an object keyed by cluster UUIDs to track the pipelines for the cluster
   * @param {Object} plugins - plugin information keyed by cluster UUIDs to count the unique plugins
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   */
  processStateResults(results, {
    clusters,
    allEphemeralIds,
    plugins
  }, monitoringClusterUuid) {
    var _results$hits2;
    const currHits = (results === null || results === void 0 ? void 0 : (_results$hits2 = results.hits) === null || _results$hits2 === void 0 ? void 0 : _results$hits2.hits) || [];
    currHits.forEach(hit => {
      var _hit$_source3, _hit$_source3$logstas, _hit$_source3$logstas2, _hit$_source3$logstas3, _hit$_source3$logstas4, _clusters$clusterUuid, _clusters$clusterUuid2, _hit$_source4, _hit$_source4$logstas;
      const clusterUuid = ((_hit$_source3 = hit._source) === null || _hit$_source3 === void 0 ? void 0 : (_hit$_source3$logstas = _hit$_source3.logstash) === null || _hit$_source3$logstas === void 0 ? void 0 : (_hit$_source3$logstas2 = _hit$_source3$logstas.pipeline) === null || _hit$_source3$logstas2 === void 0 ? void 0 : (_hit$_source3$logstas3 = _hit$_source3$logstas2.elasticsearch) === null || _hit$_source3$logstas3 === void 0 ? void 0 : (_hit$_source3$logstas4 = _hit$_source3$logstas3.cluster) === null || _hit$_source3$logstas4 === void 0 ? void 0 : _hit$_source3$logstas4.id) || monitoringClusterUuid;
      const pipelineStats = (_clusters$clusterUuid = clusters[clusterUuid]) === null || _clusters$clusterUuid === void 0 ? void 0 : (_clusters$clusterUuid2 = _clusters$clusterUuid.cluster_stats) === null || _clusters$clusterUuid2 === void 0 ? void 0 : _clusters$clusterUuid2.pipelines;
      // pipeline is defined in the mapping but contains plugin info in a reality
      const logstashStatePlugin = (_hit$_source4 = hit._source) === null || _hit$_source4 === void 0 ? void 0 : (_hit$_source4$logstas = _hit$_source4.logstash) === null || _hit$_source4$logstas === void 0 ? void 0 : _hit$_source4$logstas.pipeline;
      if (pipelineStats !== undefined && logstashStatePlugin !== undefined) {
        var _logstashStatePlugin$, _logstashStatePlugin$2, _logstashStatePlugin$3, _clusters$clusterUuid3;
        const pluginType = logstashStatePlugin === null || logstashStatePlugin === void 0 ? void 0 : (_logstashStatePlugin$ = logstashStatePlugin.plugin) === null || _logstashStatePlugin$ === void 0 ? void 0 : _logstashStatePlugin$.type;
        const pluginName = pluginType ? logstashStatePlugin === null || logstashStatePlugin === void 0 ? void 0 : (_logstashStatePlugin$2 = logstashStatePlugin.plugin) === null || _logstashStatePlugin$2 === void 0 ? void 0 : (_logstashStatePlugin$3 = _logstashStatePlugin$2[`${pluginType}`]) === null || _logstashStatePlugin$3 === void 0 ? void 0 : _logstashStatePlugin$3.name : undefined;
        if (pluginName !== undefined && pluginType !== undefined) {
          (0, _get_high_level_stats.incrementByKey)(plugins[clusterUuid], `logstash-${pluginType}-${pluginName}`);
        }
        const clusterStats = (_clusters$clusterUuid3 = clusters[clusterUuid]) === null || _clusters$clusterUuid3 === void 0 ? void 0 : _clusters$clusterUuid3.cluster_stats;
        if (clusterStats !== undefined) {
          clusterStats.plugins = (0, _get_high_level_stats.mapToList)(plugins[clusterUuid], 'name');
        }
      }
    });
  }

  /*
   * Creates a query and executes against ES to fetch agent monitoring, Logstash stats metrics
   * @param {Object} callCluster - ES client
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   */
  async fetchLogstashStats(callCluster, monitoringClusterUuid, start, end, {
    page = 0,
    ...options
  }) {
    var _results$hits3;
    const filterPath = ['hits.hits._source.cluster_uuid', 'hits.hits._source.agent.type', 'hits.hits._source.host.id', 'hits.hits._source.logstash.elasticsearch.cluster.id',
    // alias for cluster_uuid
    'hits.hits._source.logstash.node.stats.logstash'];
    const params = {
      index: _constants.INDEX_PATTERN_LOGSTASH_METRICS_NODE,
      ignore_unavailable: true,
      filter_path: filterPath,
      query: (0, _create_query.createQuery)({
        filters: [{
          bool: {
            should: [{
              term: {
                'data_stream.dataset': 'logstash.node'
              }
            }]
          }
        }, {
          range: {
            '@timestamp': {
              format: 'epoch_millis',
              gte: _moment.default.utc(start).valueOf(),
              lte: _moment.default.utc(end).valueOf()
            }
          }
        }]
      }),
      collapse: {
        field: 'host.id'
      },
      sort: [{
        '@timestamp': {
          order: 'desc',
          unmapped_type: 'long'
        }
      }],
      from: page * _logstash_monitoring.HITS_SIZE,
      size: _logstash_monitoring.HITS_SIZE
    };
    const results = await callCluster.search(params, {
      headers: {
        'X-QUERY-SOURCE': _constants.TELEMETRY_QUERY_SOURCE
      }
    });
    const hitsLength = (results === null || results === void 0 ? void 0 : (_results$hits3 = results.hits) === null || _results$hits3 === void 0 ? void 0 : _results$hits3.hits.length) || 0;
    if (hitsLength > 0) {
      // further augment the clusters object with more stats
      this.processStatsResults(results, options, monitoringClusterUuid);
    }
    return Promise.resolve();
  }

  /*
   * Creates a query and executes against ES to fetch agent monitoring, Logstash state metrics
   * @param {Object} callCluster - ES client
   * @param {string} monitoringClusterUuid - monitoring cluster UUID
   * @param {Array} hostIds - Logstash host IDs
   * @param {string} start - start timestamp
   * @param {string} end - end timestamp
   * @param {Object} options - additional processing required options
   */
  async fetchLogstashState(callCluster, hostIds, monitoringClusterUuid, start, end, {
    page = 0,
    ...options
  }) {
    const filters = [{
      bool: {
        should: [{
          term: {
            'data_stream.dataset': 'logstash.plugins'
          }
        }]
      }
    }, {
      terms: {
        'host.id': hostIds
      }
    }, {
      range: {
        '@timestamp': {
          format: 'epoch_millis',
          gte: _moment.default.utc(start).valueOf(),
          lte: _moment.default.utc(end).valueOf()
        }
      }
    }];

    // collapse by `plugin-{type}.id` to gather unique plugins pipeline is using
    for (const pluginType of _logstash_monitoring.LOGSTASH_PLUGIN_TYPES) {
      var _results$hits4;
      const params = {
        index: _constants.INDEX_PATTERN_LOGSTASH_METRICS_PLUGINS,
        ignore_unavailable: true,
        filter_path: ['hits.hits._source.logstash.pipeline'],
        query: (0, _create_query.createQuery)({
          filters
        }),
        collapse: {
          field: `logstash.pipeline.plugin.${pluginType}.id`
        },
        sort: [{
          '@timestamp': {
            order: 'desc',
            unmapped_type: 'long'
          }
        }],
        size: _logstash_monitoring.HITS_SIZE
      };
      const results = await callCluster.search(params, {
        headers: {
          'X-QUERY-SOURCE': _constants.TELEMETRY_QUERY_SOURCE
        }
      });
      const hitsLength = (results === null || results === void 0 ? void 0 : (_results$hits4 = results.hits) === null || _results$hits4 === void 0 ? void 0 : _results$hits4.hits.length) || 0;
      if (hitsLength > 0) {
        this.processStateResults(results, options, monitoringClusterUuid);
      }
    }
    return Promise.resolve();
  }
}
exports.LogstashAgentMonitoring = LogstashAgentMonitoring;