"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useBreadcrumbs = exports.useBreadcrumbContainerContext = exports.BreadcrumbContainer = void 0;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _constate = _interopRequireDefault(require("constate"));
var _public = require("@kbn/kibana-react-plugin/public");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Helper for making objects to use in a link element
const createCrumb = (url, label, testSubj, ignoreGlobalState = false) => {
  const crumb = {
    url,
    label,
    ignoreGlobalState
  };
  if (testSubj) {
    crumb.testSubj = testSubj;
  }
  return crumb;
};

// generate Elasticsearch breadcrumbs
function getElasticsearchBreadcrumbs(mainInstance) {
  const breadcrumbs = [];
  if (mainInstance.instance) {
    breadcrumbs.push(createCrumb('#/elasticsearch', 'Elasticsearch'));
    if (mainInstance.name === 'indices') {
      breadcrumbs.push(createCrumb('#/elasticsearch/indices', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.es.indicesLabel', {
        defaultMessage: 'Indices'
      }), 'breadcrumbEsIndices'));
    } else if (mainInstance.name === 'nodes') {
      breadcrumbs.push(createCrumb('#/elasticsearch/nodes', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.es.nodesLabel', {
        defaultMessage: 'Nodes'
      }), 'breadcrumbEsNodes'));
    } else if (mainInstance.name === 'ml') {
      // ML Instance (for user later)
      breadcrumbs.push(createCrumb('#/elasticsearch/ml_jobs', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.es.jobsLabel', {
        defaultMessage: 'Machine learning jobs'
      })));
    } else if (mainInstance.name === 'ccr_shard') {
      breadcrumbs.push(createCrumb('#/elasticsearch/ccr', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.es.ccrLabel', {
        defaultMessage: 'CCR'
      })));
    }
    breadcrumbs.push(createCrumb(null, mainInstance.instance));
  } else {
    // don't link to Overview when we're possibly on Overview or its sibling tabs
    breadcrumbs.push(createCrumb(null, 'Elasticsearch'));
  }
  return breadcrumbs;
}

// generate Kibana breadcrumbs
function getKibanaBreadcrumbs(mainInstance) {
  const breadcrumbs = [];
  if (mainInstance.instance) {
    breadcrumbs.push(createCrumb('#/kibana', 'Kibana'));
    breadcrumbs.push(createCrumb('#/kibana/instances', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.kibana.instancesLabel', {
      defaultMessage: 'Instances'
    })));
    breadcrumbs.push(createCrumb(null, mainInstance.instance));
  } else {
    // don't link to Overview when we're possibly on Overview or its sibling tabs
    breadcrumbs.push(createCrumb(null, 'Kibana'));
  }
  return breadcrumbs;
}

// generate Logstash breadcrumbs
function getLogstashBreadcrumbs(mainInstance) {
  const logstashLabel = _i18n.i18n.translate('xpack.monitoring.breadcrumbs.logstashLabel', {
    defaultMessage: 'Logstash'
  });
  const breadcrumbs = [];
  if (mainInstance.instance) {
    breadcrumbs.push(createCrumb('#/logstash', logstashLabel));
    if (mainInstance.name === 'nodes') {
      breadcrumbs.push(createCrumb('#/logstash/nodes', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.logstash.nodesLabel', {
        defaultMessage: 'Nodes'
      })));
    }
    breadcrumbs.push(createCrumb(null, mainInstance.instance));
  } else if (mainInstance.page === 'pipeline') {
    breadcrumbs.push(createCrumb('#/logstash', logstashLabel));
    breadcrumbs.push(createCrumb('#/logstash/pipelines', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.logstash.pipelinesLabel', {
      defaultMessage: 'Pipelines'
    })));
  } else {
    // don't link to Overview when we're possibly on Overview or its sibling tabs
    breadcrumbs.push(createCrumb(null, logstashLabel));
  }
  return breadcrumbs;
}

// generate Beats breadcrumbs
function getBeatsBreadcrumbs(mainInstance) {
  const beatsLabel = _i18n.i18n.translate('xpack.monitoring.breadcrumbs.beatsLabel', {
    defaultMessage: 'Beats'
  });
  const breadcrumbs = [];
  if (mainInstance.instance) {
    breadcrumbs.push(createCrumb('#/beats', beatsLabel));
    breadcrumbs.push(createCrumb('#/beats/beats', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.beats.instancesLabel', {
      defaultMessage: 'Instances'
    })));
    breadcrumbs.push(createCrumb(null, mainInstance.instance));
  } else {
    breadcrumbs.push(createCrumb(null, beatsLabel));
  }
  return breadcrumbs;
}

// generate Apm breadcrumbs
function getApmBreadcrumbs(mainInstance) {
  const apmLabel = _i18n.i18n.translate('xpack.monitoring.breadcrumbs.apmLabel', {
    defaultMessage: 'APM server'
  });
  const breadcrumbs = [];
  if (mainInstance.instance) {
    breadcrumbs.push(createCrumb('#/apm', apmLabel));
    breadcrumbs.push(createCrumb('#/apm/instances', _i18n.i18n.translate('xpack.monitoring.breadcrumbs.apm.instancesLabel', {
      defaultMessage: 'Instances'
    })));
    breadcrumbs.push(createCrumb(null, mainInstance.instance));
  } else {
    // don't link to Overview when we're possibly on Overview or its sibling tabs
    breadcrumbs.push(createCrumb(null, apmLabel));
  }
  return breadcrumbs;
}

// generate Enterprise Search breadcrumbs
function getEnterpriseSearchBreadcrumbs(mainInstance) {
  const entSearchLabel = _i18n.i18n.translate('xpack.monitoring.breadcrumbs.entSearchLabel', {
    defaultMessage: 'Enterprise Search'
  });
  const breadcrumbs = [];
  breadcrumbs.push(createCrumb('#/enterprise_search', entSearchLabel));
  return breadcrumbs;
}
function buildBreadcrumbs(clusterName, mainInstance) {
  const homeCrumb = _i18n.i18n.translate('xpack.monitoring.breadcrumbs.clustersLabel', {
    defaultMessage: 'Clusters'
  });
  let breadcrumbs = [createCrumb('#/home', homeCrumb, 'breadcrumbClusters', true)];
  if (!(mainInstance !== null && mainInstance !== void 0 && mainInstance.inOverview) && clusterName) {
    breadcrumbs.push(createCrumb('#/overview', clusterName));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inElasticsearch) {
    breadcrumbs = breadcrumbs.concat(getElasticsearchBreadcrumbs(mainInstance));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inKibana) {
    breadcrumbs = breadcrumbs.concat(getKibanaBreadcrumbs(mainInstance));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inLogstash) {
    breadcrumbs = breadcrumbs.concat(getLogstashBreadcrumbs(mainInstance));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inBeats) {
    breadcrumbs = breadcrumbs.concat(getBeatsBreadcrumbs(mainInstance));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inApm) {
    breadcrumbs = breadcrumbs.concat(getApmBreadcrumbs(mainInstance));
  }
  if (mainInstance !== null && mainInstance !== void 0 && mainInstance.inEnterpriseSearch) {
    breadcrumbs = breadcrumbs.concat(getEnterpriseSearchBreadcrumbs(mainInstance));
  }
  return breadcrumbs;
}
const useBreadcrumbs = ({
  history
}) => {
  const chrome = (0, _public.useKibana)().services.chrome;
  const [breadcrumbs, setBreadcrumbs] = (0, _react.useState)([]);
  const update = (0, _react.useCallback)(bcrumbs => {
    var _bcrumbs, _bcrumbs$0$href;
    if (!chrome) return;
    if (!bcrumbs) {
      var _chrome$getBreadcrumb;
      const currentBreadcrumbs = (_chrome$getBreadcrumb = chrome.getBreadcrumbs$()) === null || _chrome$getBreadcrumb === void 0 ? void 0 : _chrome$getBreadcrumb.source;
      if (currentBreadcrumbs && currentBreadcrumbs.value) {
        bcrumbs = currentBreadcrumbs.value;
      }
    }
    const globalStateStr = location.hash.split('?')[1];
    if (!((_bcrumbs = bcrumbs) !== null && _bcrumbs !== void 0 && _bcrumbs.length) || (globalStateStr === null || globalStateStr === void 0 ? void 0 : globalStateStr.indexOf('_g')) !== 0 || ((_bcrumbs$0$href = bcrumbs[0].href) === null || _bcrumbs$0$href === void 0 ? void 0 : _bcrumbs$0$href.split('?')[1]) === globalStateStr) {
      return;
    }
    bcrumbs.forEach(breadcrumb => {
      var _breadcrumb$href;
      const breadcrumbHref = (_breadcrumb$href = breadcrumb.href) === null || _breadcrumb$href === void 0 ? void 0 : _breadcrumb$href.split('?')[0];
      if (breadcrumbHref && !breadcrumb.ignoreGlobalState) {
        breadcrumb.href = `${breadcrumbHref}?${globalStateStr}`;
      }
      delete breadcrumb.ignoreGlobalState;
    });
    chrome.setBreadcrumbs(bcrumbs.slice(0));
  }, [chrome]);
  const generate = (0, _react.useCallback)((cluster, mainInstance) => {
    const crumbs = buildBreadcrumbs(cluster, mainInstance);
    setBreadcrumbs(crumbs);
    update(crumbs.map(b => ({
      text: b.label,
      href: b.url ? b.url : undefined,
      'data-test-subj': b.testSubj,
      ignoreGlobalState: b.ignoreGlobalState
    })));
  }, [setBreadcrumbs, update]);
  (0, _react.useEffect)(() => {
    history.listen((location, action) => {
      update();
    });
  }, [history, update]);
  return {
    generate,
    update,
    breadcrumbs
  };
};
exports.useBreadcrumbs = useBreadcrumbs;
const [BreadcrumbContainer, useBreadcrumbContainerContext] = (0, _constate.default)(useBreadcrumbs);
exports.useBreadcrumbContainerContext = useBreadcrumbContainerContext;
exports.BreadcrumbContainer = BreadcrumbContainer;