"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterModules = filterModules;
var _string_utils = require("../util/string_utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function filterModules(items, clauses) {
  if (clauses.length === 0) {
    return items;
  }
  // keep count of the number of matches we make as we're looping over the clauses
  // we only want to return items which match all clauses, i.e. each search term is ANDed
  const matches = items.reduce((p, module) => {
    p[module.id] = {
      module,
      count: 0
    };
    return p;
  }, {});
  clauses.forEach(c => {
    // the search term could be negated with a minus, e.g. -bananas
    const bool = c.match === 'must';
    let matchingItems = [];
    if (c.type === 'term') {
      // filter term based clauses, e.g. bananas
      // match on id, description, title
      // if the term has been negated, AND the matches
      if (bool === true) {
        matchingItems = items.filter(item => (0, _string_utils.stringMatch)(item.id, c.value) === bool || (0, _string_utils.stringMatch)(item.description, c.value) === bool || (0, _string_utils.stringMatch)(item.title, c.value) === bool);
      } else {
        matchingItems = items.filter(item => (0, _string_utils.stringMatch)(item.id, c.value) === bool && (0, _string_utils.stringMatch)(item.description, c.value) === bool && (0, _string_utils.stringMatch)(item.title, c.value) === bool);
      }
    } else {
      // filter other clauses, i.e. the tags filter
      if (c.type === 'field') {
        if (Array.isArray(c.value)) {
          // Module tags is an array of string(s) e.g. c.value => ['observability', 'logs', 'security']
          matchingItems = items.filter(item => c.value.some(filterValue => {
            var _item;
            return Array.isArray(item[c.field]) ? (_item = item[c.field]) === null || _item === void 0 ? void 0 : _item.includes(filterValue) : item[c.field] === filterValue;
          }));
        } else {
          matchingItems = items.filter(item => item[c.field] === c.value);
        }
      }
    }
    matchingItems.forEach(j => matches[j.id].count++);
  });

  // loop through the matches and return only those items which have match all the clauses
  const filtered = Object.values(matches).filter(m => (m && m.count) >= clauses.length).map(m => m.module);
  return filtered;
}