"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.mlApiProvider = mlApiProvider;
var _lodash = require("lodash");
var _app = require("../../../../common/constants/app");
var _json_schema = require("./json_schema");
var _annotations = require("./annotations");
var _data_frame_analytics = require("./data_frame_analytics");
var _filters = require("./filters");
var _results = require("./results");
var _jobs = require("./jobs");
var _saved_objects = require("./saved_objects");
var _trained_models = require("./trained_models");
var _notifications = require("./notifications");
var _inference_models = require("./inference_models");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function mlApiProvider(httpService) {
  return {
    getJobs(obj) {
      const jobId = obj && obj.jobId ? `/${obj.jobId}` : '';
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors${jobId}`,
        version: '1'
      });
    },
    getJobs$(obj) {
      const jobId = obj && obj.jobId ? `/${obj.jobId}` : '';
      return httpService.http$({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors${jobId}`,
        version: '1'
      });
    },
    getJobStats(obj) {
      const jobId = obj && obj.jobId ? `/${obj.jobId}` : '';
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors${jobId}/_stats`,
        version: '1'
      });
    },
    addJob({
      jobId,
      job
    }) {
      const body = JSON.stringify(job);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}`,
        method: 'PUT',
        body,
        version: '1'
      });
    },
    openJob({
      jobId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_open`,
        method: 'POST',
        version: '1'
      });
    },
    closeJob({
      jobId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_close`,
        method: 'POST',
        version: '1'
      });
    },
    forceCloseJob({
      jobId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_close?force=true`,
        method: 'POST',
        version: '1'
      });
    },
    deleteJob({
      jobId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}`,
        method: 'DELETE',
        version: '1'
      });
    },
    forceDeleteJob({
      jobId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}?force=true`,
        method: 'DELETE',
        version: '1'
      });
    },
    updateJob({
      jobId,
      job
    }) {
      const body = JSON.stringify(job);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_update`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    estimateBucketSpan(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/validate/estimate_bucket_span`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    validateJob(payload) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/validate/job`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    validateDatafeedPreview(payload) {
      const body = JSON.stringify(payload);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/validate/datafeed_preview`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    validateCardinality$(job) {
      const body = JSON.stringify(job);
      return httpService.http$({
        path: `${_app.ML_INTERNAL_BASE_PATH}/validate/cardinality`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    getDatafeeds(obj) {
      const datafeedId = obj && obj.datafeedId ? `/${obj.datafeedId}` : '';
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds${datafeedId}`,
        version: '1'
      });
    },
    getDatafeedStats(obj) {
      const datafeedId = obj && obj.datafeedId ? `/${obj.datafeedId}` : '';
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds${datafeedId}/_stats`,
        version: '1'
      });
    },
    addDatafeed({
      datafeedId,
      datafeedConfig
    }) {
      const body = JSON.stringify(datafeedConfig);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}`,
        method: 'PUT',
        body,
        version: '1'
      });
    },
    updateDatafeed({
      datafeedId,
      datafeedConfig
    }) {
      const body = JSON.stringify(datafeedConfig);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}/_update`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    deleteDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}`,
        method: 'DELETE',
        version: '1'
      });
    },
    forceDeleteDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}?force=true`,
        method: 'DELETE',
        version: '1'
      });
    },
    startDatafeed({
      datafeedId,
      start,
      end
    }) {
      // if the end timestamp is a number, add one ms to it to make it
      // inclusive of the end of the data
      if ((0, _lodash.isNumber)(end)) {
        end++;
      }
      const body = JSON.stringify({
        ...(start !== undefined ? {
          start
        } : {}),
        ...(end !== undefined ? {
          end
        } : {})
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}/_start`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    stopDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}/_stop`,
        method: 'POST',
        version: '1'
      });
    },
    forceStopDatafeed({
      datafeedId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}/_stop?force=true`,
        method: 'POST',
        version: '1'
      });
    },
    datafeedPreview({
      datafeedId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/datafeeds/${datafeedId}/_preview`,
        method: 'GET',
        version: '1'
      });
    },
    forecast({
      jobId,
      duration,
      neverExpires
    }) {
      const body = JSON.stringify({
        ...(duration !== undefined ? {
          duration
        } : {}),
        ...(neverExpires === true ? {
          expires_in: '0'
        } : {})
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_forecast`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    deleteForecast({
      jobId,
      forecastId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/_forecast/${forecastId}`,
        method: 'DELETE',
        version: '1'
      });
    },
    overallBuckets({
      jobId,
      topN,
      bucketSpan,
      start,
      end,
      overallScore
    }) {
      const body = JSON.stringify({
        topN,
        bucketSpan,
        start,
        end,
        ...(overallScore ? {
          overall_score: overallScore
        } : {})
      });

      // Max permitted job_id is 64 characters, so we can fit around 30 jobs per request
      const maxJobsPerRequest = 30;
      return Promise.all((0, _lodash.chunk)(jobId, maxJobsPerRequest).map(jobIdsChunk => {
        return httpService.http({
          path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobIdsChunk.join(',')}/results/overall_buckets`,
          method: 'POST',
          body,
          version: '1'
        });
      })).then(responses => {
        // Merge responses
        return responses.reduce((acc, response) => {
          acc.count += response.count;
          acc.overall_buckets.push(...response.overall_buckets);
          return acc;
        }, {
          count: 0,
          overall_buckets: []
        });
      });
    },
    hasPrivileges(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/_has_privileges`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    checkMlCapabilities() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/ml_capabilities`,
        method: 'GET',
        version: '1'
      });
    },
    checkIndicesExists({
      indices
    }) {
      const body = JSON.stringify({
        indices
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/index_exists`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    recognizeIndex({
      indexPatternTitle,
      filter
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/recognize/${indexPatternTitle}`,
        method: 'GET',
        version: '1',
        query: {
          filter: filter === null || filter === void 0 ? void 0 : filter.join(',')
        }
      });
    },
    recognizeModule({
      moduleId,
      size
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/recognize_by_module/${moduleId}`,
        method: 'GET',
        version: '1',
        query: {
          size
        }
      });
    },
    listDataRecognizerModules(filter) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/get_module`,
        method: 'GET',
        version: '1',
        query: {
          filter: filter === null || filter === void 0 ? void 0 : filter.join(',')
        }
      });
    },
    getDataRecognizerModule(params) {
      const {
        moduleId,
        filter
      } = params || {};
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/get_module/${moduleId !== null && moduleId !== void 0 ? moduleId : ''}`,
        method: 'GET',
        version: '1',
        query: {
          filter: filter === null || filter === void 0 ? void 0 : filter.join(',')
        }
      });
    },
    dataRecognizerModuleJobsExist({
      moduleId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/jobs_exist/${moduleId}`,
        method: 'GET',
        version: '1'
      });
    },
    setupDataRecognizerConfig({
      moduleId,
      prefix,
      groups,
      indexPatternName,
      query,
      useDedicatedIndex,
      startDatafeed,
      start,
      end,
      jobOverrides,
      estimateModelMemory
    }) {
      const body = JSON.stringify({
        prefix,
        groups,
        indexPatternName,
        query,
        useDedicatedIndex,
        startDatafeed,
        start,
        end,
        jobOverrides,
        estimateModelMemory
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/modules/setup/${moduleId}`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    getVisualizerFieldHistograms({
      indexPattern,
      query,
      fields,
      samplerShardSize,
      runtimeMappings
    }) {
      const body = JSON.stringify({
        query,
        fields,
        samplerShardSize,
        runtimeMappings
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/data_visualizer/get_field_histograms/${indexPattern}`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    /**
     * Gets a list of calendars
     * @param obj
     * @returns {Promise<MlCalendar[]>}
     */
    calendars(obj) {
      const {
        calendarId,
        calendarIds
      } = obj || {};
      let calendarIdsPathComponent = '';
      if (calendarId) {
        calendarIdsPathComponent = `/${calendarId}`;
      } else if (calendarIds) {
        calendarIdsPathComponent = `/${calendarIds.join(',')}`;
      }
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/calendars${calendarIdsPathComponent}`,
        method: 'GET',
        version: '1'
      });
    },
    addCalendar(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/calendars`,
        method: 'PUT',
        body,
        version: '1'
      });
    },
    updateCalendar(obj) {
      const calendarId = obj && obj.calendarId ? `/${obj.calendarId}` : '';
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/calendars${calendarId}`,
        method: 'PUT',
        body,
        version: '1'
      });
    },
    deleteCalendar({
      calendarId
    }) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/calendars/${calendarId}`,
        method: 'DELETE',
        version: '1'
      });
    },
    mlNodeCount() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/ml_node_count`,
        method: 'GET',
        version: '1'
      });
    },
    mlInfo() {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/info`,
        method: 'GET',
        version: '1'
      });
    },
    calculateModelMemoryLimit$({
      datafeedConfig,
      analysisConfig,
      indexPattern,
      query,
      timeFieldName,
      earliestMs,
      latestMs
    }) {
      const body = JSON.stringify({
        datafeedConfig,
        analysisConfig,
        indexPattern,
        query,
        timeFieldName,
        earliestMs,
        latestMs
      });
      return httpService.http$({
        path: `${_app.ML_INTERNAL_BASE_PATH}/validate/calculate_model_memory_limit`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    getCardinalityOfFields({
      index,
      fieldNames,
      query,
      timeFieldName,
      earliestMs,
      latestMs
    }) {
      const body = JSON.stringify({
        index,
        fieldNames,
        query,
        timeFieldName,
        earliestMs,
        latestMs
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/fields_service/field_cardinality`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    getTimeFieldRange({
      index,
      timeFieldName,
      query,
      runtimeMappings,
      indicesOptions,
      allowFutureTime
    }) {
      const body = JSON.stringify({
        index,
        timeFieldName,
        query,
        runtimeMappings,
        indicesOptions,
        allowFutureTime
      });
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/fields_service/time_field_range`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    esSearch(obj) {
      const body = JSON.stringify(obj);
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/es_search`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    esSearch$(obj) {
      const body = JSON.stringify(obj);
      return httpService.http$({
        path: `${_app.ML_INTERNAL_BASE_PATH}/es_search`,
        method: 'POST',
        body,
        version: '1'
      });
    },
    getIndices() {
      const tempBasePath = '/api';
      return httpService.http({
        path: `${tempBasePath}/index_management/indices`,
        method: 'GET',
        version: '1'
      });
    },
    getModelSnapshots(jobId, snapshotId) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/model_snapshots${snapshotId !== undefined ? `/${snapshotId}` : ''}`,
        version: '1'
      });
    },
    updateModelSnapshot(jobId, snapshotId, body) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/model_snapshots/${snapshotId}/_update`,
        method: 'POST',
        body: JSON.stringify(body),
        version: '1'
      });
    },
    deleteModelSnapshot(jobId, snapshotId) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/anomaly_detectors/${jobId}/model_snapshots/${snapshotId}`,
        method: 'DELETE',
        version: '1'
      });
    },
    reindexWithPipeline(pipelineName, sourceIndex, destinationIndex) {
      return httpService.http({
        path: `${_app.ML_INTERNAL_BASE_PATH}/reindex_with_pipeline`,
        method: 'POST',
        body: JSON.stringify({
          source: {
            index: sourceIndex
          },
          dest: {
            index: destinationIndex,
            pipeline: pipelineName
          }
        }),
        version: '1'
      });
    },
    annotations: (0, _annotations.annotationsApiProvider)(httpService),
    dataFrameAnalytics: (0, _data_frame_analytics.dataFrameAnalyticsApiProvider)(httpService),
    filters: (0, _filters.filtersApiProvider)(httpService),
    results: (0, _results.resultsApiProvider)(httpService),
    jobs: (0, _jobs.jobsApiProvider)(httpService),
    savedObjects: (0, _saved_objects.savedObjectsApiProvider)(httpService),
    trainedModels: (0, _trained_models.trainedModelsApiProvider)(httpService),
    inferenceModels: (0, _inference_models.inferenceModelsApiProvider)(httpService),
    notifications: (0, _notifications.notificationsProvider)(httpService),
    jsonSchema: (0, _json_schema.jsonSchemaProvider)(httpService)
  };
}