"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TrainedModelsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _mlTrainedModelsUtils = require("@kbn/ml-trained-models-utils");
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _trained_models = require("../../../common/types/trained_models");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DOWNLOAD_POLL_INTERVAL = 3000;
class TrainedModelsService {
  constructor(trainedModelsApiService) {
    (0, _defineProperty2.default)(this, "_reloadSubject$", new _rxjs.Subject());
    (0, _defineProperty2.default)(this, "_modelItems$", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "downloadStatus$", new _rxjs.BehaviorSubject({}));
    (0, _defineProperty2.default)(this, "pollingSubscription", void 0);
    (0, _defineProperty2.default)(this, "abortedDownloads", new Set());
    (0, _defineProperty2.default)(this, "downloadStatusFetchInProgress", false);
    (0, _defineProperty2.default)(this, "setScheduledDeployments", void 0);
    (0, _defineProperty2.default)(this, "displayErrorToast", void 0);
    (0, _defineProperty2.default)(this, "displaySuccessToast", void 0);
    (0, _defineProperty2.default)(this, "subscription", void 0);
    (0, _defineProperty2.default)(this, "_scheduledDeployments$", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "destroySubscription", void 0);
    (0, _defineProperty2.default)(this, "_isLoading$", new _rxjs.BehaviorSubject(true));
    (0, _defineProperty2.default)(this, "savedObjectsApiService", void 0);
    (0, _defineProperty2.default)(this, "canManageSpacesAndSavedObjects", void 0);
    (0, _defineProperty2.default)(this, "isInitialized", false);
    (0, _defineProperty2.default)(this, "telemetryService", void 0);
    (0, _defineProperty2.default)(this, "deploymentParamsMapper", void 0);
    (0, _defineProperty2.default)(this, "uiInitiatedDownloads", new Set());
    (0, _defineProperty2.default)(this, "isLoading$", this._isLoading$.pipe((0, _rxjs.distinctUntilChanged)()));
    (0, _defineProperty2.default)(this, "modelItems$", this._modelItems$.pipe((0, _rxjs.distinctUntilChanged)(_lodash.isEqual)));
    this.trainedModelsApiService = trainedModelsApiService;
  }
  init({
    scheduledDeployments$,
    setScheduledDeployments,
    displayErrorToast,
    displaySuccessToast,
    savedObjectsApiService,
    canManageSpacesAndSavedObjects,
    telemetryService,
    deploymentParamsMapper
  }) {
    // Always cancel any pending destroy when trying to initialize
    if (this.destroySubscription) {
      this.destroySubscription.unsubscribe();
      this.destroySubscription = undefined;
    }
    if (this.isInitialized) return;
    this.subscription = new _rxjs.Subscription();
    this.isInitialized = true;
    this.canManageSpacesAndSavedObjects = canManageSpacesAndSavedObjects;
    this.deploymentParamsMapper = deploymentParamsMapper;
    this.setScheduledDeployments = setScheduledDeployments;
    this._scheduledDeployments$ = scheduledDeployments$;
    this.displayErrorToast = displayErrorToast;
    this.displaySuccessToast = displaySuccessToast;
    this.savedObjectsApiService = savedObjectsApiService;
    this.telemetryService = telemetryService;
    this.setupFetchingSubscription();
    this.setupDeploymentSubscription();
  }
  get scheduledDeployments$() {
    return this._scheduledDeployments$;
  }
  get scheduledDeployments() {
    return this._scheduledDeployments$.getValue();
  }
  get modelItems() {
    return this._modelItems$.getValue();
  }
  get isLoading() {
    return this._isLoading$.getValue();
  }
  fetchModels() {
    const timestamp = Date.now();
    this._reloadSubject$.next(timestamp);
  }
  startModelDeployment(modelId, deploymentParams) {
    var _this$setScheduledDep;
    const newDeployment = {
      modelId,
      ...deploymentParams
    };
    const currentDeployments = this._scheduledDeployments$.getValue();
    (_this$setScheduledDep = this.setScheduledDeployments) === null || _this$setScheduledDep === void 0 ? void 0 : _this$setScheduledDep.call(this, [...currentDeployments, newDeployment]);
  }
  downloadModel(modelId) {
    this._isLoading$.next(true);
    this.uiInitiatedDownloads.add(modelId);
    (0, _rxjs.from)(this.trainedModelsApiService.installElasticTrainedModelConfig(modelId)).pipe((0, _rxjs.finalize)(() => {
      this.fetchModels();
    })).subscribe({
      error: error => {
        var _this$displayErrorToa;
        (_this$displayErrorToa = this.displayErrorToast) === null || _this$displayErrorToa === void 0 ? void 0 : _this$displayErrorToa.call(this, error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.downloadFailed', {
          defaultMessage: 'Failed to download "{modelId}"',
          values: {
            modelId
          }
        }));
        this.telemetryService.trackTrainedModelsModelDownload({
          model_id: modelId,
          result: 'failure'
        });
        this.uiInitiatedDownloads.delete(modelId);
      }
    });
  }
  updateModelDeployment(modelId, config) {
    const apiParams = this.deploymentParamsMapper.mapUiToApiDeploymentParams(modelId, config);
    (0, _rxjs.from)(this.trainedModelsApiService.updateModelDeployment(modelId, config.deploymentId, this.getUpdateModelAllocationParams(apiParams))).pipe((0, _rxjs.tap)({
      next: () => {
        var _this$displaySuccessT;
        (_this$displaySuccessT = this.displaySuccessToast) === null || _this$displaySuccessT === void 0 ? void 0 : _this$displaySuccessT.call(this, {
          title: _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.updateSuccess', {
            defaultMessage: 'Deployment updated'
          }),
          text: _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.updateSuccessText', {
            defaultMessage: '"{deploymentId}" has been updated successfully.',
            values: {
              deploymentId: config.deploymentId
            }
          })
        });
      },
      error: error => {
        var _this$displayErrorToa2;
        (_this$displayErrorToa2 = this.displayErrorToast) === null || _this$displayErrorToa2 === void 0 ? void 0 : _this$displayErrorToa2.call(this, error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.updateFailed', {
          defaultMessage: 'Failed to update "{deploymentId}"',
          values: {
            deploymentId: config.deploymentId
          }
        }));
      }
    }), (0, _rxjs.map)(() => ({
      success: true
    })), (0, _rxjs.catchError)(() => (0, _rxjs.of)({
      success: false
    })), (0, _rxjs.finalize)(() => {
      this.fetchModels();
    })).subscribe(result => {
      var _apiParams$adaptiveAl, _apiParams$adaptiveAl2;
      this.telemetryService.trackTrainedModelsDeploymentUpdated({
        adaptive_resources: config.adaptiveResources,
        model_id: modelId,
        optimized: config.optimized,
        vcpu_usage: config.vCPUUsage,
        max_number_of_allocations: (_apiParams$adaptiveAl = apiParams.adaptiveAllocationsParams) === null || _apiParams$adaptiveAl === void 0 ? void 0 : _apiParams$adaptiveAl.max_number_of_allocations,
        min_number_of_allocations: (_apiParams$adaptiveAl2 = apiParams.adaptiveAllocationsParams) === null || _apiParams$adaptiveAl2 === void 0 ? void 0 : _apiParams$adaptiveAl2.min_number_of_allocations,
        number_of_allocations: apiParams.deploymentParams.number_of_allocations,
        threads_per_allocation: apiParams.deploymentParams.threads_per_allocation,
        result: result.success ? 'success' : 'failure'
      });
    });
  }
  async deleteModels(modelIds, options) {
    const results = await Promise.allSettled(modelIds.map(modelId => this.trainedModelsApiService.deleteTrainedModel({
      modelId,
      options
    }).then(() => this.removeScheduledDeployments({
      modelId
    }))));

    // Handle failures
    const failures = results.map((result, index) => ({
      modelId: modelIds[index],
      failed: result.status === 'rejected',
      error: result.status === 'rejected' ? result.reason : null
    })).filter(r => r.failed);
    if (failures.length > 0) {
      var _this$displayErrorToa3;
      const failedModelIds = failures.map(f => f.modelId).join(', ');
      (_this$displayErrorToa3 = this.displayErrorToast) === null || _this$displayErrorToa3 === void 0 ? void 0 : _this$displayErrorToa3.call(this, failures[0].error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.fetchDeletionErrorTitle', {
        defaultMessage: '{modelsCount, plural, one {Model} other {Models}} deletion failed',
        values: {
          modelsCount: failures.length
        }
      }), undefined, failedModelIds);
    }
    this.fetchModels();
  }
  stopModelDeployment(modelId, deploymentIds, options) {
    (0, _rxjs.from)(this.trainedModelsApiService.stopModelAllocation(modelId, deploymentIds, options)).pipe((0, _rxjs.finalize)(() => {
      this.fetchModels();
    })).subscribe({
      error: error => {
        var _this$displayErrorToa4;
        (_this$displayErrorToa4 = this.displayErrorToast) === null || _this$displayErrorToa4 === void 0 ? void 0 : _this$displayErrorToa4.call(this, error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.stopFailed', {
          defaultMessage: 'Failed to stop "{deploymentIds}"',
          values: {
            deploymentIds: deploymentIds.join(', ')
          }
        }));
      }
    });
  }
  getModel(modelId) {
    return this.modelItems.find(item => item.model_id === modelId);
  }
  getModel$(modelId) {
    return this._modelItems$.pipe((0, _rxjs.map)(items => items.find(item => item.model_id === modelId)), (0, _rxjs.distinctUntilChanged)(_lodash.isEqual));
  }

  /** Removes scheduled deployments for a model */
  removeScheduledDeployments({
    modelId,
    deploymentId
  }) {
    var _this$setScheduledDep2;
    let updated = this._scheduledDeployments$.getValue();

    // If removing by modelId, abort currently downloading model and filter all deployments for that model.
    if (modelId) {
      const model = this.getModel(modelId);
      const isDownloading = model && (0, _trained_models.isBaseNLPModelItem)(model) && model.state === _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADING;
      if (isDownloading) {
        this.abortDownload(modelId);
      }
      updated = updated.filter(d => d.modelId !== modelId);
    }

    // If removing by deploymentId, filter deployments matching that ID.
    if (deploymentId) {
      updated = updated.filter(d => d.deploymentId !== deploymentId);
    }
    (_this$setScheduledDep2 = this.setScheduledDeployments) === null || _this$setScheduledDep2 === void 0 ? void 0 : _this$setScheduledDep2.call(this, updated);
  }
  isModelReadyForDeployment(model) {
    if (!model || !(0, _trained_models.isBaseNLPModelItem)(model)) {
      return false;
    }
    return model.state === _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADED || model.state === _mlTrainedModelsUtils.MODEL_STATE.STARTED;
  }
  setDeployingStateForModel(modelId) {
    const currentModels = this.modelItems;
    const updatedModels = currentModels.map(model => (0, _trained_models.isBaseNLPModelItem)(model) && model.model_id === modelId ? {
      ...model,
      state: _mlTrainedModelsUtils.MODEL_STATE.STARTING
    } : model);
    this._modelItems$.next(updatedModels);
  }
  abortDownload(modelId) {
    this.abortedDownloads.add(modelId);
  }
  mergeModelItems(items, spaces) {
    const existingItems = this._modelItems$.getValue();
    return items.map(item => {
      var _spaces$item$model_id;
      const previous = existingItems.find(m => m.model_id === item.model_id);
      const merged = {
        ...item,
        spaces: (_spaces$item$model_id = spaces[item.model_id]) !== null && _spaces$item$model_id !== void 0 ? _spaces$item$model_id : []
      };
      if (!previous || !(0, _trained_models.isBaseNLPModelItem)(previous) || !(0, _trained_models.isBaseNLPModelItem)(item)) {
        return merged;
      }

      // Preserve "DOWNLOADING" state and the accompanying progress if still in progress
      if (previous.state === _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADING) {
        return {
          ...merged,
          state: previous.state,
          downloadState: previous.downloadState
        };
      }

      // If was "STARTING" and there's still a scheduled deployment, keep it in "STARTING"
      if (previous.state === _mlTrainedModelsUtils.MODEL_STATE.STARTING && this.scheduledDeployments.some(d => d.modelId === item.model_id) && item.state !== _mlTrainedModelsUtils.MODEL_STATE.STARTED) {
        return {
          ...merged,
          state: previous.state
        };
      }
      return merged;
    });
  }
  setupFetchingSubscription() {
    this.subscription.add(this._reloadSubject$.pipe((0, _rxjs.tap)(() => this._isLoading$.next(true)), (0, _rxjs.debounceTime)(100), (0, _rxjs.switchMap)(() => {
      const modelsList$ = (0, _rxjs.from)(this.trainedModelsApiService.getTrainedModelsList()).pipe((0, _rxjs.catchError)(error => {
        var _this$displayErrorToa5;
        (_this$displayErrorToa5 = this.displayErrorToast) === null || _this$displayErrorToa5 === void 0 ? void 0 : _this$displayErrorToa5.call(this, error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.fetchFailedErrorMessage', {
          defaultMessage: 'Error loading trained models'
        }));
        return (0, _rxjs.of)([]);
      }));
      const spaces$ = this.canManageSpacesAndSavedObjects ? (0, _rxjs.from)(this.savedObjectsApiService.trainedModelsSpaces()).pipe((0, _rxjs.catchError)(() => (0, _rxjs.of)({})), (0, _rxjs.map)(spaces => 'trainedModels' in spaces ? spaces.trainedModels : {})) : (0, _rxjs.of)({});
      return (0, _rxjs.forkJoin)([modelsList$, spaces$]).pipe((0, _rxjs.finalize)(() => this._isLoading$.next(false)));
    })).subscribe(([items, spaces]) => {
      const updatedItems = this.mergeModelItems(items, spaces);
      this._modelItems$.next(updatedItems);
      this.startDownloadStatusPolling();
    }));
  }
  setupDeploymentSubscription() {
    this.subscription.add(this._scheduledDeployments$.pipe((0, _rxjs.filter)(deployments => deployments.length > 0), (0, _rxjs.tap)(() => this.fetchModels()), (0, _rxjs.switchMap)(deployments => this._isLoading$.pipe((0, _rxjs.filter)(isLoading => !isLoading), (0, _rxjs.take)(1), (0, _rxjs.map)(() => deployments))),
    // Check if the model is already deployed and remove it from the scheduled deployments if so
    (0, _rxjs.switchMap)(deployments => {
      const filteredDeployments = deployments.filter(deployment => {
        const model = this.modelItems.find(m => m.model_id === deployment.modelId);
        return !(model && this.isModelAlreadyDeployed(model, deployment));
      });
      return (0, _rxjs.of)(filteredDeployments).pipe((0, _rxjs.tap)(filtered => {
        if (!(0, _lodash.isEqual)(deployments, filtered)) {
          var _this$setScheduledDep3;
          (_this$setScheduledDep3 = this.setScheduledDeployments) === null || _this$setScheduledDep3 === void 0 ? void 0 : _this$setScheduledDep3.call(this, filtered);
        }
      }), (0, _rxjs.filter)(filtered => (0, _lodash.isEqual)(deployments, filtered)) // Only proceed if no changes were made
      );
    }), (0, _rxjs.switchMap)(deployments => (0, _rxjs.merge)(...deployments.map(deployment => this.handleDeployment$(deployment))))).subscribe());
  }
  handleDeployment$(deployment) {
    return (0, _rxjs.of)(deployment).pipe(
    // Wait for the model to be ready for deployment (downloaded or started)
    (0, _rxjs.switchMap)(() => {
      return this.waitForModelReady(deployment.modelId);
    }), (0, _rxjs.tap)(() => this.setDeployingStateForModel(deployment.modelId)), (0, _rxjs.map)(() => this.deploymentParamsMapper.mapUiToApiDeploymentParams(deployment.modelId, deployment)), (0, _rxjs.exhaustMap)(apiParams => {
      return (0, _rxjs.firstValueFrom)(this.trainedModelsApiService.startModelAllocation(apiParams).pipe((0, _rxjs.tap)({
        next: () => {
          var _this$displaySuccessT2;
          (_this$displaySuccessT2 = this.displaySuccessToast) === null || _this$displaySuccessT2 === void 0 ? void 0 : _this$displaySuccessT2.call(this, {
            title: _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.startSuccess', {
              defaultMessage: 'Deployment started'
            }),
            text: _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.startSuccessText', {
              defaultMessage: '"{deploymentId}" has started successfully.',
              values: {
                deploymentId: deployment.deploymentId
              }
            })
          });
        }
      }), (0, _rxjs.map)(() => ({
        success: true
      })), (0, _rxjs.catchError)(error => {
        var _this$displayErrorToa6;
        (_this$displayErrorToa6 = this.displayErrorToast) === null || _this$displayErrorToa6 === void 0 ? void 0 : _this$displayErrorToa6.call(this, error, _i18n.i18n.translate('xpack.ml.trainedModels.modelsList.startFailed', {
          defaultMessage: 'Failed to start "{deploymentId}"',
          values: {
            deploymentId: deployment.deploymentId
          }
        }));

        // Return observable to allow stream to continue
        return (0, _rxjs.of)({
          success: false
        });
      }), (0, _rxjs.tap)(result => {
        var _apiParams$adaptiveAl3, _apiParams$adaptiveAl4;
        this.telemetryService.trackTrainedModelsDeploymentCreated({
          model_id: apiParams.modelId,
          optimized: deployment.optimized,
          adaptive_resources: deployment.adaptiveResources,
          vcpu_usage: deployment.vCPUUsage,
          number_of_allocations: apiParams.deploymentParams.number_of_allocations,
          threads_per_allocation: apiParams.deploymentParams.threads_per_allocation,
          min_number_of_allocations: (_apiParams$adaptiveAl3 = apiParams.adaptiveAllocationsParams) === null || _apiParams$adaptiveAl3 === void 0 ? void 0 : _apiParams$adaptiveAl3.min_number_of_allocations,
          max_number_of_allocations: (_apiParams$adaptiveAl4 = apiParams.adaptiveAllocationsParams) === null || _apiParams$adaptiveAl4 === void 0 ? void 0 : _apiParams$adaptiveAl4.max_number_of_allocations,
          result: result.success ? 'success' : 'failure'
        });
      }), (0, _rxjs.finalize)(() => {
        this.removeScheduledDeployments({
          deploymentId: deployment.deploymentId
        });
        // Manually update the BehaviorSubject to ensure proper cleanup
        // if user navigates away, as localStorage hook won't be available to handle updates
        const updatedDeployments = this._scheduledDeployments$.getValue().filter(d => d.modelId !== deployment.modelId);
        this._scheduledDeployments$.next(updatedDeployments);
        this.fetchModels();
      })));
    }));
  }
  getUpdateModelAllocationParams(apiParams) {
    return apiParams.adaptiveAllocationsParams ? {
      adaptive_allocations: apiParams.adaptiveAllocationsParams
    } : {
      number_of_allocations: apiParams.deploymentParams.number_of_allocations,
      adaptive_allocations: {
        enabled: false
      }
    };
  }
  isModelAlreadyDeployed(model, deployment) {
    return !!(model && (0, _trained_models.isNLPModelItem)(model) && (model.deployment_ids.includes(deployment.deploymentId) || model.state === _mlTrainedModelsUtils.MODEL_STATE.STARTING));
  }
  waitForModelReady(modelId) {
    return this.getModel$(modelId).pipe((0, _rxjs.filter)(model => this.isModelReadyForDeployment(model)), (0, _rxjs.take)(1));
  }

  /**
   * The polling logic is the single source of truth for whether the model
   * is still in-progress downloading. If we see an item is no longer in the
   * returned statuses, that means it’s finished or aborted, so remove the
   * "downloading" operation in activeOperations (if present).
   */
  startDownloadStatusPolling() {
    if (this.downloadStatusFetchInProgress) return;
    this.stopPolling();
    const downloadInProgress = new Set();
    this.downloadStatusFetchInProgress = true;
    this.pollingSubscription = (0, _rxjs.timer)(0, DOWNLOAD_POLL_INTERVAL).pipe((0, _rxjs.takeWhile)(() => this.downloadStatusFetchInProgress), (0, _rxjs.switchMap)(() => this.trainedModelsApiService.getModelsDownloadStatus()), (0, _rxjs.distinctUntilChanged)((prev, curr) => (0, _lodash.isEqual)(prev, curr)), (0, _rxjs.withLatestFrom)(this._modelItems$)).subscribe({
      next: ([downloadStatus, currentItems]) => {
        const updatedItems = currentItems.map(item => {
          if (!(0, _trained_models.isBaseNLPModelItem)(item)) return item;

          /* Unfortunately, model download status does not report 100% download state, only from 1 to 99. Hence, there might be 3 cases
           * 1. Model is not downloaded at all
           * 2. Model download was in progress and finished
           * 3. Model download was in progress and aborted
           */
          if (downloadStatus[item.model_id]) {
            downloadInProgress.add(item.model_id);
            return {
              ...item,
              state: _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADING,
              downloadState: downloadStatus[item.model_id]
            };
          } else {
            // Not in 'downloadStatus' => either done or aborted
            const newItem = {
              ...item
            };
            delete newItem.downloadState;
            if (this.abortedDownloads.has(item.model_id)) {
              // Aborted
              this.abortedDownloads.delete(item.model_id);
              newItem.state = _mlTrainedModelsUtils.MODEL_STATE.NOT_DOWNLOADED;
              if (this.uiInitiatedDownloads.has(item.model_id)) {
                this.telemetryService.trackTrainedModelsModelDownload({
                  model_id: item.model_id,
                  result: 'cancelled'
                });
                this.uiInitiatedDownloads.delete(item.model_id);
              }
            } else if (downloadInProgress.has(item.model_id) || !item.state) {
              // Finished downloading
              newItem.state = _mlTrainedModelsUtils.MODEL_STATE.DOWNLOADED;

              // Only track success if the model was downloading
              if (downloadInProgress.has(item.model_id) && this.uiInitiatedDownloads.has(item.model_id)) {
                this.telemetryService.trackTrainedModelsModelDownload({
                  model_id: item.model_id,
                  result: 'success'
                });
                this.uiInitiatedDownloads.delete(item.model_id);
              }
            }
            downloadInProgress.delete(item.model_id);
            return newItem;
          }
        });
        this._modelItems$.next(updatedItems);
        this.downloadStatus$.next(downloadStatus);
        Object.keys(downloadStatus).forEach(modelId => {
          if (downloadStatus[modelId]) {
            downloadInProgress.add(modelId);
          }
        });
        if (Object.keys(downloadStatus).length === 0 && downloadInProgress.size === 0) {
          this.stopPolling();
          this.downloadStatusFetchInProgress = false;
        }
      },
      error: error => {
        this.stopPolling();
        this.downloadStatusFetchInProgress = false;
        this.uiInitiatedDownloads.clear();
      }
    });
  }
  stopPolling() {
    if (this.pollingSubscription) {
      this.pollingSubscription.unsubscribe();
    }
    this.downloadStatusFetchInProgress = false;
  }
  cleanupService() {
    // Clear operation state
    this.abortedDownloads.clear();
    this.downloadStatusFetchInProgress = false;

    // Clear subscriptions
    if (this.pollingSubscription) {
      this.pollingSubscription.unsubscribe();
    }
    if (this.subscription) {
      this.subscription.unsubscribe();
    }

    // Reset behavior subjects to initial values
    this._modelItems$.next([]);
    this.downloadStatus$.next({});
    this._scheduledDeployments$.next([]);

    // Clear callbacks
    this.setScheduledDeployments = undefined;
    this.displayErrorToast = undefined;
    this.displaySuccessToast = undefined;

    // Reset initialization flag
    this.isInitialized = false;
  }
  destroy() {
    // Cancel any pending destroy
    if (this.destroySubscription) {
      this.destroySubscription.unsubscribe();
      this.destroySubscription = undefined;
    }

    // Wait for scheduled deployments to be empty before cleaning up
    this.destroySubscription = this._scheduledDeployments$.pipe((0, _rxjs.filter)(deployments => deployments.length === 0), (0, _rxjs.take)(1)).subscribe({
      complete: () => {
        this.cleanupService();
        this.destroySubscription = undefined;
      }
    });
  }
}
exports.TrainedModelsService = TrainedModelsService;