"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.JobRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _states = require("../../../../../../common/constants/states");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const REFRESH_INTERVAL_MS = 250;
const NODE_ASSIGNMENT_CHECK_REFRESH_INTERVAL_MS = 2000;
const TARGET_PROGRESS_DELTA = 2;
const REFRESH_RATE_ADJUSTMENT_DELAY_MS = 2000;
class JobRunner {
  constructor(jobCreator) {
    (0, _defineProperty2.default)(this, "_mlApi", void 0);
    (0, _defineProperty2.default)(this, "_jobId", void 0);
    (0, _defineProperty2.default)(this, "_datafeedId", void 0);
    (0, _defineProperty2.default)(this, "_start", 0);
    (0, _defineProperty2.default)(this, "_end", 0);
    (0, _defineProperty2.default)(this, "_datafeedState", _states.DATAFEED_STATE.STOPPED);
    (0, _defineProperty2.default)(this, "_refreshInterval", REFRESH_INTERVAL_MS);
    (0, _defineProperty2.default)(this, "_progress$", void 0);
    (0, _defineProperty2.default)(this, "_percentageComplete", 0);
    (0, _defineProperty2.default)(this, "_stopRefreshPoll", void 0);
    (0, _defineProperty2.default)(this, "_subscribers", void 0);
    (0, _defineProperty2.default)(this, "_datafeedStartTime", 0);
    (0, _defineProperty2.default)(this, "_performRefreshRateAdjustment", false);
    (0, _defineProperty2.default)(this, "_jobAssignedToNode", false);
    (0, _defineProperty2.default)(this, "_jobAssignedToNode$", void 0);
    this._mlApi = jobCreator.mlApi;
    this._jobId = jobCreator.jobId;
    this._datafeedId = jobCreator.datafeedId;
    this._start = jobCreator.start;
    this._end = jobCreator.end;
    this._percentageComplete = 0;
    this._stopRefreshPoll = jobCreator.stopAllRefreshPolls;
    this._progress$ = new _rxjs.BehaviorSubject(this._percentageComplete);
    this._jobAssignedToNode$ = new _rxjs.BehaviorSubject(this._jobAssignedToNode);
    this._subscribers = jobCreator.subscribers;
  }
  get datafeedState() {
    return this._datafeedState;
  }
  set refreshInterval(v) {
    this._refreshInterval = v;
  }
  resetInterval() {
    this._refreshInterval = REFRESH_INTERVAL_MS;
  }
  async openJob() {
    try {
      const {
        node
      } = await this._mlApi.openJob({
        jobId: this._jobId
      });
      this._jobAssignedToNode = node !== undefined && node.length > 0;
      this._jobAssignedToNode$.next(this._jobAssignedToNode);
    } catch (error) {
      throw error;
    }
  }

  // start the datafeed and then start polling for progress
  // the complete percentage is added to an observable
  // so all pre-subscribed listeners can follow along.
  async _startDatafeed(start, end, pollProgress) {
    try {
      this._datafeedStartTime = Date.now();
      // link the _subscribers list from the JobCreator
      // to the progress BehaviorSubject.
      const subscriptions = pollProgress === true ? this._subscribers.map(s => this._progress$.subscribe(s)) : [];
      await this.openJob();
      const {
        started
      } = await this._mlApi.startDatafeed({
        datafeedId: this._datafeedId,
        start,
        end
      });
      this._datafeedState = _states.DATAFEED_STATE.STARTED;
      this._percentageComplete = 0;
      const checkProgress = async () => {
        const {
          isRunning,
          progress: prog,
          isJobClosed
        } = await this.getProgress();

        // if the progress has reached 100% but the job is still running,
        // dial the progress back to 99 to avoid any post creation buttons from
        // appearing as they only rely on the progress.
        const progress = prog === 100 && (isRunning === true || isJobClosed === false) ? prog - 1 : prog;
        this._adjustRefreshInterval(progress);
        this._percentageComplete = progress;
        this._progress$.next(this._percentageComplete);
        if ((isRunning === true || isJobClosed === false) && this._stopRefreshPoll.stop === false) {
          setTimeout(async () => {
            if (this._stopRefreshPoll.stop === false) {
              await checkProgress();
            }
          }, this._refreshInterval);
        } else {
          // job has finished running, set progress to 100%
          // it may be lower than 100 on completion as the progress
          // is calculated based on latest_record_timestamp which may be earlier
          // than the end date supplied to the datafeed
          this._progress$.next(100);
          // unsubscribe everyone
          subscriptions.forEach(s => s.unsubscribe());
        }
      };
      const checkJobIsAssigned = async () => {
        this._jobAssignedToNode = await this._isJobAssigned();
        this._jobAssignedToNode$.next(this._jobAssignedToNode);
        if (this._jobAssignedToNode === true) {
          await checkProgress();
        } else {
          setTimeout(async () => {
            if (this._stopRefreshPoll.stop === false) {
              await checkJobIsAssigned();
            }
          }, NODE_ASSIGNMENT_CHECK_REFRESH_INTERVAL_MS);
        }
      };
      // wait for the first check to run and then return success.
      // all subsequent checks will update the observable
      if (pollProgress === true) {
        if (this._jobAssignedToNode === true) {
          await checkProgress();
        } else {
          await checkJobIsAssigned();
        }
      }
      return started;
    } catch (error) {
      throw error;
    }
  }
  _adjustRefreshInterval(progress) {
    if (this._performRefreshRateAdjustment === false) {
      // for the first couple of seconds of the job running, don't
      // adjust the refresh interval
      const timeDeltaMs = Date.now() - this._datafeedStartTime;
      if (timeDeltaMs > REFRESH_RATE_ADJUSTMENT_DELAY_MS) {
        this._performRefreshRateAdjustment = true;
      } else {
        return;
      }
    }
    const progressDelta = progress - this._percentageComplete;
    if (progressDelta !== 0) {
      // adjust the refresh interval so that it produces a change in percentage
      // that is close to the target
      this._refreshInterval = Math.floor(this._refreshInterval * (TARGET_PROGRESS_DELTA / progressDelta));

      // don't let the interval fall below the initial default.
      if (this._refreshInterval < REFRESH_INTERVAL_MS) {
        this._refreshInterval = REFRESH_INTERVAL_MS;
      }
    }
  }
  async _isJobAssigned() {
    const {
      jobs
    } = await this._mlApi.getJobStats({
      jobId: this._jobId
    });
    return jobs.length > 0 && jobs[0].node !== undefined;
  }
  async startDatafeed() {
    return await this._startDatafeed(this._start, this._end, true);
  }
  async startDatafeedInRealTime(continueJob) {
    // if continuing a job, set the start to be the end date
    const start = continueJob ? this._end : this._start;
    return await this._startDatafeed(start, undefined, false);
  }
  async getProgress() {
    return await this._mlApi.jobs.getLookBackProgress(this._jobId, this._start, this._end);
  }
  subscribeToProgress(func) {
    this._progress$.subscribe(func);
  }
  async isRunning() {
    const {
      isRunning
    } = await this.getProgress();
    return isRunning;
  }
  isJobAssignedToNode() {
    return this._jobAssignedToNode;
  }
  subscribeToJobAssignment(func) {
    return this._jobAssignedToNode$.subscribe(func);
  }
}
exports.JobRunner = JobRunner;