"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createMetricByFieldLookup = createMetricByFieldLookup;
exports.metricsToApiOptions = metricsToApiOptions;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/*
They key part of this file is the function 'createFieldLookup'.

The metrics_explorer endpoint expects a list of the metrics to use, like this:
[
  { field: 'some.metric.field', aggregation: 'avg' },
  { field: 'some.other.metric.field', aggregation: 'min' },
]

The API then responds with a series, which is a list of rows (buckets from a date_histogram 
aggregation), where each bucket has this format:
{ metric_0: 99, metric_1: 88 }

For each metric in the request, a key like metric_X is defined, and the number used is the order in
which the metric appeared in the request. So if the metric for 'some.metric.field' is first, it'll
be mapped to metric_0, but if the code changes and it is now second, it will be mapped to metric_1.

This makes the code that consumes the API response fragile to such re-ordering, the types and 
functions in this file are used to reduce this fragility and allowing consuming code to reference
the metrics by their field names instead. 
The returned metricByField object, handles the translation from field name to "index name".
For example, in the transform function passed to useInfrastructureNodeMetrics it can be used
to find a field metric like this:
row[metricByField['some.field']]

If the endpoint where to change its return format to:
{ 'some.metric.field': 99, 'some.other.metric.field': 88 }
Then this code would no longer be needed.
*/

// MetricsQueryOptions wraps the basic things needed to build the query for metrics
// sourceFilter is used to define filter to get only relevant docs that contain metrics info
// e.g: { 'source.module': 'system' }

// The input to this generic type is a (union) string type that defines all the fields we want to
// request metrics for. This input type serves as something like a "source of truth" for which
// fields are being used. The resulting MetricsMap and metricByField helper ensures a type safe
// usage of the metrics data returned from the API.

// MetricsMap uses an object type to ensure each field gets defined.
// This type only ensures that the MetricsMap is defined in a way that the key matches the field
// it uses
// { 'some-field: { field: 'some-field', aggregation: 'whatever' } }

function metricsToApiOptions(metricsQueryOptions, filterClauseDsl) {
  const metrics = Object.values(metricsQueryOptions.metricsMap);
  const options = {
    aggregation: 'avg',
    groupBy: metricsQueryOptions.groupByField,
    metrics,
    filterQuery: JSON.stringify(buildFilterClause(metricsQueryOptions.sourceFilter, filterClauseDsl))
  };
  return {
    options
  };
}
function buildFilterClause(sourceFilter, filterClauseDsl) {
  return {
    bool: {
      filter: !!filterClauseDsl ? [sourceFilter, filterClauseDsl] : [sourceFilter]
    }
  };
}
function createMetricByFieldLookup(metricMap) {
  const fields = Object.keys(metricMap);
  const metrics = Object.values(metricMap);
  const setMetricIndexToField = (acc, field) => {
    return {
      ...acc,
      [field]: fieldToMetricIndex(field, metrics)
    };
  };
  return fields.reduce(setMetricIndexToField, {});
}
function fieldToMetricIndex(field, metrics) {
  const index = metrics.findIndex(metric => metric.field === field);
  if (index === -1) {
    throw new Error('Failed to find index for field ' + field);
  }
  return `metric_${index}`;
}