"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.GeoJsonFileSource = void 0;
var _vector_source = require("../vector_source");
var _constants = require("../../../../common/constants");
var _get_feature_collection_bounds = require("../../util/get_feature_collection_bounds");
var _inline_field = require("../../fields/inline_field");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function convertToFeatureCollection(geoJson) {
  if (!geoJson) {
    return _constants.EMPTY_FEATURE_COLLECTION;
  }
  if (geoJson.type === 'FeatureCollection') {
    return geoJson;
  }
  if (geoJson.type === 'Feature') {
    return {
      type: 'FeatureCollection',
      features: [geoJson]
    };
  }
  return _constants.EMPTY_FEATURE_COLLECTION;
}
class GeoJsonFileSource extends _vector_source.AbstractVectorSource {
  static createDescriptor(descriptor) {
    return {
      type: _constants.SOURCE_TYPES.GEOJSON_FILE,
      __featureCollection: convertToFeatureCollection(descriptor.__featureCollection),
      __fields: descriptor.__fields || [],
      areResultsTrimmed: descriptor.areResultsTrimmed !== undefined ? descriptor.areResultsTrimmed : false,
      tooltipContent: descriptor.tooltipContent ? descriptor.tooltipContent : null,
      name: descriptor.name || 'Features'
    };
  }
  constructor(descriptor) {
    const normalizedDescriptor = GeoJsonFileSource.createDescriptor(descriptor);
    super(normalizedDescriptor);
  }
  _getFieldDescriptors() {
    const fields = this._descriptor.__fields;
    return fields ? fields : [];
  }
  _createField(fieldDescriptor) {
    return new _inline_field.InlineField({
      fieldName: fieldDescriptor.name,
      source: this,
      origin: _constants.FIELD_ORIGIN.SOURCE,
      dataType: fieldDescriptor.type
    });
  }
  async getFields() {
    return this._getFieldDescriptors().map(fieldDescriptor => {
      return this._createField(fieldDescriptor);
    });
  }
  getFieldByName(fieldName) {
    const fieldDescriptor = this._getFieldDescriptors().find(findFieldDescriptor => {
      return findFieldDescriptor.name === fieldName;
    });
    return fieldDescriptor ? this._createField(fieldDescriptor) : null;
  }
  isBoundsAware() {
    return true;
  }
  async getBoundsForFilters(boundsFilters, registerCancelCallback) {
    const featureCollection = this.getFeatureCollection();
    return (0, _get_feature_collection_bounds.getFeatureCollectionBounds)(featureCollection, false);
  }
  async getGeoJsonWithMeta() {
    return {
      data: this.getFeatureCollection(),
      meta: {}
    };
  }
  async getDisplayName() {
    return this._descriptor.name;
  }
  hasTooltipProperties() {
    return true;
  }
  getSourceStatus() {
    return {
      tooltipContent: this._descriptor.tooltipContent,
      areResultsTrimmed: this._descriptor.areResultsTrimmed
    };
  }
  getFeatureCollection() {
    return this._descriptor.__featureCollection;
  }
}
exports.GeoJsonFileSource = GeoJsonFileSource;