"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMultiTermsAggregateRoute = exports.doSearch = void 0;
var _configSchema = require("@kbn/config-schema");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerMultiTermsAggregateRoute = (router, logger) => {
  router.versioned.get({
    access: 'internal',
    path: _constants.MULTI_TERMS_AGGREGATE_ROUTE
  }).addVersion({
    version: '1',
    security: {
      authz: {
        requiredPrivileges: ['securitySolution']
      }
    },
    validate: {
      request: {
        query: _configSchema.schema.object({
          index: _configSchema.schema.string(),
          query: _configSchema.schema.string(),
          countBy: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_RESOURCE_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_NAMESPACE), _configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_NAME), _configSchema.schema.literal(_constants.CLOUD_INSTANCE_NAME), _configSchema.schema.literal(_constants.CONTAINER_IMAGE_NAME), _configSchema.schema.literal(_constants.ENTRY_LEADER_ENTITY_ID)])),
          groupBys: _configSchema.schema.arrayOf(_configSchema.schema.object({
            field: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_RESOURCE_ID), _configSchema.schema.literal(_constants.ORCHESTRATOR_NAMESPACE), _configSchema.schema.literal(_constants.ORCHESTRATOR_CLUSTER_NAME), _configSchema.schema.literal(_constants.CLOUD_INSTANCE_NAME), _configSchema.schema.literal(_constants.CONTAINER_IMAGE_NAME), _configSchema.schema.literal(_constants.ENTRY_LEADER_USER_ID), _configSchema.schema.literal(_constants.ENTRY_LEADER_INTERACTIVE)]),
            missing: _configSchema.schema.maybe(_configSchema.schema.string())
          }), {
            defaultValue: []
          }),
          page: _configSchema.schema.number({
            max: 10000,
            min: 0
          }),
          perPage: _configSchema.schema.maybe(_configSchema.schema.number({
            max: 100,
            min: 1
          }))
        })
      }
    }
  }, async (context, request, response) => {
    const client = (await context.core).elasticsearch.client.asCurrentUser;
    const {
      query,
      countBy,
      groupBys,
      page,
      perPage,
      index
    } = request.query;
    try {
      const body = await doSearch(client, index, query, groupBys, page, perPage, countBy);
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      logger.error(`Failed to fetch k8s multi_terms_aggregate: ${err}`);
      return response.customError({
        body: {
          message: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.registerMultiTermsAggregateRoute = registerMultiTermsAggregateRoute;
const doSearch = async (client, index, query, groupBys, page,
// zero based
perPage = _constants.AGGREGATE_PAGE_SIZE, countBy) => {
  var _search$aggregations;
  const queryDSL = JSON.parse(query);
  const countByAggs = countBy ? {
    count_by_aggs: {
      cardinality: {
        field: countBy
      }
    },
    count_alerts: {
      cardinality: {
        field: countBy
      }
    }
  } : undefined;
  const search = await client.search({
    index: [index],
    body: {
      query: queryDSL,
      size: 0,
      aggs: {
        custom_agg: {
          multi_terms: {
            terms: groupBys
          },
          aggs: {
            ...countByAggs,
            bucket_sort: {
              bucket_sort: {
                size: perPage + 1,
                // check if there's a "next page"
                from: perPage * page
              }
            }
          }
        }
      }
    }
  });
  const agg = (_search$aggregations = search.aggregations) === null || _search$aggregations === void 0 ? void 0 : _search$aggregations.custom_agg;
  const buckets = (agg === null || agg === void 0 ? void 0 : agg.buckets) || [];
  const hasNextPage = buckets.length > perPage;
  if (hasNextPage) {
    buckets.pop();
  }
  return {
    buckets,
    hasNextPage
  };
};
exports.doSearch = doSearch;