"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createChatCompleteCallbackApi = createChatCompleteCallbackApi;
var _inferenceCommon = require("@kbn/inference-common");
var _rxjs = require("rxjs");
var _inferenceTracing = require("@kbn/inference-tracing");
var _lodash = require("lodash");
var _adapters = require("./adapters");
var _utils = require("./utils");
var _retry_with_exponential_backoff = require("../../common/utils/retry_with_exponential_backoff");
var _error_retry_filter = require("../../common/utils/error_retry_filter");
var _anonymize_messages = require("./anonymization/anonymize_messages");
var _deanonymize_message = require("./anonymization/deanonymize_message");
var _add_anonymization_instruction = require("./anonymization/add_anonymization_instruction");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createChatCompleteCallbackApi({
  request,
  actions,
  logger,
  anonymizationRulesPromise,
  regexWorker,
  esClient
}) {
  return ({
    connectorId,
    abortSignal,
    stream,
    maxRetries = 3,
    retryConfiguration = {
      retryOn: 'all'
    }
  }, callback) => {
    const inference$ = (0, _rxjs.defer)(() => (0, _rxjs.forkJoin)({
      executor: (0, _rxjs.from)((0, _utils.getInferenceExecutor)({
        connectorId,
        request,
        actions
      })),
      anonymizationRules: (0, _rxjs.from)(anonymizationRulesPromise)
    })).pipe((0, _rxjs.switchMap)(({
      executor,
      anonymizationRules
    }) => {
      const {
        system,
        messages: givenMessages,
        functionCalling,
        metadata,
        modelName,
        temperature,
        toolChoice,
        tools
      } = callback(executor);
      const messages = givenMessages.map(message => {
        // remove empty toolCalls array, spec doesn't like it
        if (message.role === _inferenceCommon.MessageRole.Assistant && message.toolCalls !== undefined && message.toolCalls.length === 0) {
          return (0, _lodash.omit)(message, 'toolCalls');
        }
        return message;
      });
      return (0, _rxjs.from)((0, _anonymize_messages.anonymizeMessages)({
        system,
        messages,
        anonymizationRules,
        regexWorker,
        esClient
      })).pipe((0, _rxjs.switchMap)(anonymization => {
        const connector = executor.getConnector();
        const connectorType = connector.type;
        const inferenceAdapter = (0, _adapters.getInferenceAdapter)(connectorType);
        if (!inferenceAdapter) {
          return (0, _rxjs.throwError)(() => (0, _inferenceCommon.createInferenceRequestError)(`Adapter for type ${connectorType} not implemented`, 400));
        }
        const systemWithAnonymizationInstructions = anonymization.system ? (0, _add_anonymization_instruction.addAnonymizationInstruction)(anonymization.system, anonymizationRules) : system;
        return (0, _inferenceTracing.withChatCompleteSpan)({
          system: systemWithAnonymizationInstructions,
          messages: anonymization.messages,
          tools,
          toolChoice,
          model: {
            family: (0, _inferenceCommon.getConnectorFamily)(connector),
            provider: (0, _inferenceCommon.getConnectorProvider)(connector)
          },
          ...(metadata === null || metadata === void 0 ? void 0 : metadata.attributes)
        }, () => {
          return inferenceAdapter.chatComplete({
            system: systemWithAnonymizationInstructions,
            executor,
            messages: anonymization.messages,
            toolChoice,
            tools,
            temperature,
            logger,
            functionCalling,
            modelName,
            abortSignal,
            metadata
          }).pipe((0, _utils.chunksIntoMessage)({
            toolOptions: {
              toolChoice,
              tools
            },
            logger
          }));
        }).pipe((0, _deanonymize_message.deanonymizeMessage)(anonymization));
      }));
    })).pipe((0, _retry_with_exponential_backoff.retryWithExponentialBackoff)({
      maxRetry: maxRetries,
      backoffMultiplier: retryConfiguration.backoffMultiplier,
      initialDelay: retryConfiguration.initialDelay,
      errorFilter: (0, _error_retry_filter.getRetryFilter)(retryConfiguration.retryOn)
    }), abortSignal ? (0, _utils.handleCancellation)(abortSignal) : _rxjs.identity);
    if (stream) {
      return inference$.pipe((0, _rxjs.share)());
    } else {
      return (0, _utils.streamToResponse)(inference$);
    }
  };
}