"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.correctTimespanLiterals = void 0;
var _esqlAst = require("@kbn/esql-ast");
var _typeguards = require("../typeguards");
var _timespan = require("../ast_tools/timespan");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Correct timespan literal grammar mistakes, and returns the list of corrections that got applied.
 *
 * E.g.
 * `DATE_TRUNC("YEAR", @timestamp)` => `DATE_TRUNC(1 year, @timestamp)`
 * `BUCKET(@timestamp, "1 week")` => `BUCKET(@timestamp, 1 week)`
 *
 */
const correctTimespanLiterals = query => {
  const corrections = [];
  _esqlAst.Walker.walk(query, {
    visitFunction: node => {
      if ((0, _typeguards.isDateTruncFunctionNode)(node)) {
        corrections.push(...checkDateTrunc(node));
      }
      if ((0, _typeguards.isBucketFunctionNode)(node)) {
        corrections.push(...checkBucket(node));
      }
    }
  });
  return corrections;
};
exports.correctTimespanLiterals = correctTimespanLiterals;
function checkDateTrunc(node) {
  if (node.args.length !== 2) {
    return [];
  }
  const firstArg = node.args[0];
  if ((0, _typeguards.isStringLiteralNode)(firstArg) && (0, _timespan.isTimespanString)(firstArg.value)) {
    const replacement = (0, _timespan.stringToTimespanLiteral)(firstArg.value);
    node.args[0] = replacement;
    const correction = {
      type: 'string_as_timespan_literal',
      node,
      description: `Replaced string literal with timespan literal in DATE_TRUNC function at position ${node.location.min}`
    };
    return [correction];
  }
  return [];
}
function checkBucket(node) {
  // only checking the 2 args version - e.g. BUCKET(hire_date, 1 week)
  if (node.args.length !== 2) {
    return [];
  }
  const secondArg = node.args[1];
  if ((0, _typeguards.isStringLiteralNode)(secondArg) && (0, _timespan.isTimespanString)(secondArg.value)) {
    const replacement = (0, _timespan.stringToTimespanLiteral)(secondArg.value);
    node.args[1] = replacement;
    const correction = {
      type: 'string_as_timespan_literal',
      node,
      description: `Replaced string literal with timespan literal in BUCKET function at position ${node.location.min}`
    };
    return [correction];
  }
  return [];
}