"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReferencedIndexTemplateMeta = exports.registerGetDatastreams = void 0;
var _configSchema = require("@kbn/config-schema");
var _ = require("..");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const paramsSchema = _configSchema.schema.object({
  name: _configSchema.schema.string()
});
async function getDatastreamsForComponentTemplate(esClient, name) {
  const {
    component_templates: componentTemplates
  } = await esClient.cluster.getComponentTemplate({
    name
  });
  if (!componentTemplates.find(componentTemplate => componentTemplate.name === name)) {
    return [];
  }
  const {
    index_templates: indexTemplates
  } = await esClient.indices.getIndexTemplate();
  const datastreamNames = indexTemplates.filter(indexTemplate => {
    var _indexTemplate$index_, _indexTemplate$index_2;
    return (_indexTemplate$index_ = indexTemplate.index_template) === null || _indexTemplate$index_ === void 0 ? void 0 : (_indexTemplate$index_2 = _indexTemplate$index_.composed_of) === null || _indexTemplate$index_2 === void 0 ? void 0 : _indexTemplate$index_2.includes(name);
  }).map(indexTemplate => indexTemplate.index_template.index_patterns).flat().join(',');
  if (datastreamNames.length < 0) {
    return [];
  }
  const {
    data_streams: dataStreams
  } = await esClient.indices.getDataStream({
    name: datastreamNames
  });
  return dataStreams;
}
const registerGetDatastreams = ({
  router,
  lib: {
    handleEsError
  }
}) => {
  router.get({
    path: (0, _.addBasePath)('/component_templates/{name}/datastreams'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      params: paramsSchema
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      name
    } = request.params;
    try {
      const dataStreams = await getDatastreamsForComponentTemplate(client.asCurrentUser, name);
      return response.ok({
        body: {
          data_streams: dataStreams.map(ds => ds.name)
        }
      });
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
};
exports.registerGetDatastreams = registerGetDatastreams;
const registerReferencedIndexTemplateMeta = ({
  router,
  lib: {
    handleEsError
  }
}) => {
  router.get({
    path: (0, _.addBasePath)('/component_templates/{name}/referenced_index_template_meta'),
    security: {
      authz: {
        enabled: false,
        reason: 'Relies on es client for authorization'
      }
    },
    validate: {
      params: paramsSchema
    }
  }, async (context, request, response) => {
    const {
      client
    } = (await context.core).elasticsearch;
    const {
      name
    } = request.params;
    try {
      const {
        index_templates: indexTemplates
      } = await client.asCurrentUser.indices.getIndexTemplate();
      const result = indexTemplates.filter(indexTemplate => {
        var _indexTemplate$index_3, _indexTemplate$index_4;
        return (_indexTemplate$index_3 = indexTemplate.index_template) === null || _indexTemplate$index_3 === void 0 ? void 0 : (_indexTemplate$index_4 = _indexTemplate$index_3.composed_of) === null || _indexTemplate$index_4 === void 0 ? void 0 : _indexTemplate$index_4.includes(name);
      });

      // We should always match against the first result which should yield
      // the index template we are after.
      if (result[0]) {
        return response.ok({
          body: result[0].index_template._meta
        });
      }
      return response.notFound();
    } catch (error) {
      return handleEsError({
        error,
        response
      });
    }
  });
};
exports.registerReferencedIndexTemplateMeta = registerReferencedIndexTemplateMeta;