"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.updateEsAssetReferences = exports.optimisticallyAddEsAssetReferences = void 0;
var _server = require("@kbn/core/server");
var _pRetry = _interopRequireDefault(require("p-retry"));
var _lodash = require("lodash");
var _constants = require("../../../constants");
var _audit_logging = require("../../audit_logging");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Utility function for updating the installed_es field of a package
 */
const updateEsAssetReferences = async (savedObjectsClient, pkgName, currentAssets, {
  assetsToAdd = [],
  assetsToRemove = [],
  refresh = false
}) => {
  const withAssetsRemoved = currentAssets.filter(({
    type,
    id
  }) => {
    if (assetsToRemove.some(({
      type: removeType,
      id: removeId
    }) => removeType === type && removeId === id)) {
      return false;
    }
    return true;
  });
  const deduplicatedAssets = (0, _lodash.uniqBy)([...withAssetsRemoved, ...assetsToAdd], ({
    type,
    id
  }) => `${type}-${id}`);
  _audit_logging.auditLoggingService.writeCustomSoAuditLog({
    action: 'update',
    id: pkgName,
    savedObjectType: _constants.PACKAGES_SAVED_OBJECT_TYPE
  });
  const {
    attributes: {
      installed_es: updatedAssets
    }
  } =
  // Because Kibana assets are installed in parallel with ES assets with refresh: false, we almost always run into an
  // issue that causes a conflict error due to this issue: https://github.com/elastic/kibana/issues/126240. This is safe
  // to retry constantly until it succeeds to optimize this critical user journey path as much as possible.
  await (0, _pRetry.default)(() => savedObjectsClient.update(_constants.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
    installed_es: deduplicatedAssets
  }, {
    refresh
  }),
  // Use a lower number of retries for ES assets since they're installed in serial and can only conflict with
  // the single Kibana update call.
  {
    retries: 5
  });
  return updatedAssets !== null && updatedAssets !== void 0 ? updatedAssets : [];
};
/**
 * Utility function for adding assets the installed_es field of a package
 * uses optimistic concurrency control to prevent missed updates
 */
exports.updateEsAssetReferences = updateEsAssetReferences;
const optimisticallyAddEsAssetReferences = async (savedObjectsClient, pkgName, assetsToAdd, esIndexPatterns) => {
  const addEsAssets = async () => {
    var _so$attributes$instal, _so$attributes$es_ind;
    // TODO: Should this be replaced by a `get()` call from epm/get.ts?
    const so = await savedObjectsClient.get(_constants.PACKAGES_SAVED_OBJECT_TYPE, pkgName);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'get',
      id: pkgName,
      savedObjectType: _constants.PACKAGES_SAVED_OBJECT_TYPE
    });
    const installedEs = (_so$attributes$instal = so.attributes.installed_es) !== null && _so$attributes$instal !== void 0 ? _so$attributes$instal : [];
    const deduplicatedAssets = (0, _lodash.uniqBy)([...installedEs, ...assetsToAdd], ({
      type,
      id
    }) => `${type}-${id}`);
    const deduplicatedIndexPatterns = Object.assign({}, (_so$attributes$es_ind = so.attributes.es_index_patterns) !== null && _so$attributes$es_ind !== void 0 ? _so$attributes$es_ind : {}, esIndexPatterns);
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id: pkgName,
      savedObjectType: _constants.PACKAGES_SAVED_OBJECT_TYPE
    });
    const {
      attributes: {
        installed_es: updatedAssets
      }
    } = await savedObjectsClient.update(_constants.PACKAGES_SAVED_OBJECT_TYPE, pkgName, {
      installed_es: deduplicatedAssets,
      es_index_patterns: deduplicatedIndexPatterns
    }, {
      version: so.version
    });
    return updatedAssets !== null && updatedAssets !== void 0 ? updatedAssets : [];
  };
  const onlyRetryConflictErrors = err => {
    if (!_server.SavedObjectsErrorHelpers.isConflictError(err)) {
      throw err;
    }
  };
  return (0, _pRetry.default)(addEsAssets, {
    retries: 10,
    onFailedAttempt: onlyRetryConflictErrors
  });
};
exports.optimisticallyAddEsAssetReferences = optimisticallyAddEsAssetReferences;