"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAgentLogsTopErrors = getAgentLogsTopErrors;
var _lodash = require("lodash");
var _constants = require("../../common/constants");
var _services = require("../services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_LOGS_DATA = {
  agent_logs_top_errors: [],
  fleet_server_logs_top_errors: []
};
async function getAgentLogsTopErrors(esClient) {
  if (!esClient) {
    return DEFAULT_LOGS_DATA;
  }
  try {
    const queryTopMessages = index => esClient.search({
      index,
      size: 100,
      _source: ['message'],
      query: {
        bool: {
          filter: [{
            terms: {
              _tier: _constants.DATA_TIERS
            }
          }, {
            term: {
              'log.level': 'error'
            }
          }, {
            range: {
              '@timestamp': {
                gte: 'now-1h'
              }
            }
          }]
        }
      }
    });
    const getTopErrors = resp => {
      var _resp$hits$hits;
      const counts = ((_resp$hits$hits = resp === null || resp === void 0 ? void 0 : resp.hits.hits) !== null && _resp$hits$hits !== void 0 ? _resp$hits$hits : []).reduce((acc, curr) => {
        if (!acc[curr._source.message]) {
          acc[curr._source.message] = 0;
        }
        acc[curr._source.message]++;
        return acc;
      }, {});
      const top3 = (0, _lodash.sortBy)(Object.entries(counts).map(([key, value]) => ({
        key,
        value
      })), 'value').slice(0, 3).reverse();
      return top3.map(({
        key,
        value
      }) => key);
    };
    const agentResponse = await queryTopMessages('logs-elastic_agent-*');
    const fleetServerResponse = await queryTopMessages('logs-elastic_agent.fleet_server-*');
    return {
      agent_logs_top_errors: getTopErrors(agentResponse),
      fleet_server_logs_top_errors: getTopErrors(fleetServerResponse)
    };
  } catch (error) {
    if (error.statusCode === 404) {
      _services.appContextService.getLogger().debug('Index pattern logs-elastic_agent* does not exist yet.');
    } else {
      throw error;
    }
    return DEFAULT_LOGS_DATA;
  }
}