"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getCategorizationField = getCategorizationField;
exports.mapRecommendedQueriesFromExtensions = exports.getRecommendedQueriesTemplatesFromExtensions = exports.getRecommendedQueriesSuggestionsFromStaticTemplates = void 0;
var _templates = require("./templates");
var _constants = require("../../shared/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getRecommendedQueriesSuggestionsFromStaticTemplates = async (getFieldsByType, fromCommand = '') => {
  const [fieldSuggestions, textFieldSuggestions] = await Promise.all([getFieldsByType(['date'], [], {
    openSuggestions: true
  }),
  // get text fields separately to avoid mixing them with date fields
  getFieldsByType(['text'], [], {
    openSuggestions: true
  })]);
  let timeField = '';
  let categorizationField = '';
  if (fieldSuggestions.length) {
    var _fieldSuggestions$fin;
    timeField = (fieldSuggestions === null || fieldSuggestions === void 0 ? void 0 : (_fieldSuggestions$fin = fieldSuggestions.find(field => field.label === '@timestamp')) === null || _fieldSuggestions$fin === void 0 ? void 0 : _fieldSuggestions$fin.label) || fieldSuggestions[0].label;
  }
  if (textFieldSuggestions.length) {
    categorizationField = getCategorizationField(textFieldSuggestions.map(field => field.label));
  }
  const recommendedQueries = (0, _templates.getRecommendedQueries)({
    fromCommand,
    timeField,
    categorizationField
  });
  const suggestions = recommendedQueries.map(query => {
    var _query$sortText;
    return {
      label: query.label,
      text: query.queryString,
      kind: 'Issue',
      detail: query.description,
      sortText: (_query$sortText = query === null || query === void 0 ? void 0 : query.sortText) !== null && _query$sortText !== void 0 ? _query$sortText : 'E'
    };
  });
  return suggestions;
};

/**
 * This function maps the recommended queries from the extensions to the autocomplete suggestions.
 * @param recommendedQueriesExtensions, the recommended queries extensions to map
 * @returns SuggestionRawDefinition[], the mapped suggestions
 */
exports.getRecommendedQueriesSuggestionsFromStaticTemplates = getRecommendedQueriesSuggestionsFromStaticTemplates;
const mapRecommendedQueriesFromExtensions = recommendedQueriesExtensions => {
  const suggestions = recommendedQueriesExtensions.map(extension => {
    var _extension$name;
    return {
      label: extension.name,
      text: extension.query,
      detail: (_extension$name = extension.name) !== null && _extension$name !== void 0 ? _extension$name : '',
      ...(extension.description ? {
        documentation: {
          value: extension.description
        }
      } : {}),
      kind: 'Issue',
      sortText: 'D'
    };
  });
  return suggestions;
};

/**
 * This function extracts the templates from the recommended queries extensions.
 * The templates are the recommended queries without the source command (FROM).
 * This is useful for showing the templates in the autocomplete suggestions when the users have already typed the FROM command.
 * @param recommendedQueriesExtensions, the recommended queries extensions to extract the templates from
 * @returns SuggestionRawDefinition[], the templates extracted from the recommended queries extensions
 */
exports.mapRecommendedQueriesFromExtensions = mapRecommendedQueriesFromExtensions;
const getRecommendedQueriesTemplatesFromExtensions = recommendedQueriesExtensions => {
  if (!recommendedQueriesExtensions || !recommendedQueriesExtensions.length) {
    return [];
  }

  // the templates are the recommended queries without the source command (FROM)
  const recommendedQueriesTemplates = recommendedQueriesExtensions.map(recommendedQuery => {
    var _recommendedQuery$nam;
    const queryParts = recommendedQuery.query.split('|');
    // remove the first part (the FROM command)
    return {
      label: recommendedQuery.name,
      text: `|${queryParts.slice(1).join('|')}`,
      detail: (_recommendedQuery$nam = recommendedQuery.name) !== null && _recommendedQuery$nam !== void 0 ? _recommendedQuery$nam : '',
      ...(recommendedQuery.description ? {
        documentation: {
          value: recommendedQuery.description
        }
      } : {}),
      kind: 'Issue',
      sortText: 'D'
    };
  });
  return recommendedQueriesTemplates;
};

/**
 * This function returns the categorization field from the list of fields.
 * It checks for the presence of 'message', 'error.message', or 'event.original' in that order.
 * If none of these fields are present, it returns the first field from the list,
 * Assumes text fields have been passed in the `fields` array.
 *
 * This function is a duplicate of the one in src/platform/packages/shared/kbn-aiops-utils.
 * It is included here to avoid build errors due to bazel
 *
 * TODO: Remove this function once the bazel issue is resolved.
 *
 * @param fields, the list of fields to check
 * @returns string | undefined, the categorization field if found, otherwise undefined
 */
exports.getRecommendedQueriesTemplatesFromExtensions = getRecommendedQueriesTemplatesFromExtensions;
function getCategorizationField(fields) {
  var _filteredFields$;
  const fieldPriority = ['message', 'error.message', 'event.original'];
  const fieldSet = new Set(fields);
  for (const field of fieldPriority) {
    if (fieldSet.has(field)) {
      return field;
    }
  }

  // Filter out metadata fields
  const filteredFields = fields.filter(field => !_constants.METADATA_FIELDS.includes(field));
  return (_filteredFields$ = filteredFields[0]) !== null && _filteredFields$ !== void 0 ? _filteredFields$ : undefined;
}