"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.installBuiltInEntityDefinitions = installBuiltInEntityDefinitions;
exports.installEntityDefinition = installEntityDefinition;
exports.installationInProgress = void 0;
exports.reinstallEntityDefinition = reinstallEntityDefinition;
var _semver = _interopRequireDefault(require("semver"));
var _generate_component_id = require("./helpers/generate_component_id");
var _create_and_install_ingest_pipeline = require("./create_and_install_ingest_pipeline");
var _create_and_install_transform = require("./create_and_install_transform");
var _validate_transform_ids = require("./transform/validate_transform_ids");
var _delete_entity_definition = require("./delete_entity_definition");
var _delete_ingest_pipeline = require("./delete_ingest_pipeline");
var _find_entity_definition = require("./find_entity_definition");
var _save_entity_definition = require("./save_entity_definition");
var _manage_index_templates = require("../manage_index_templates");
var _entity_id_conflict_error = require("./errors/entity_id_conflict_error");
var _entity_not_found = require("./errors/entity_not_found");
var _merge_definition_update = require("./helpers/merge_definition_update");
var _stop_transforms = require("./stop_transforms");
var _delete_transforms = require("./delete_transforms");
var _delete_index = require("./delete_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// install an entity definition from scratch with all its required components
// after verifying that the definition id is valid and available.
// attempt to remove all installed components if the installation fails.
async function installEntityDefinition({
  esClient,
  soClient,
  definition,
  logger
}) {
  (0, _validate_transform_ids.validateDefinitionCanCreateValidTransformIds)(definition);
  if (await (0, _save_entity_definition.entityDefinitionExists)(soClient, definition.id)) {
    throw new _entity_id_conflict_error.EntityIdConflict(`Entity definition [${definition.id}] already exists.`, definition);
  }
  try {
    const entityDefinition = await (0, _save_entity_definition.saveEntityDefinition)(soClient, {
      ...definition,
      installStatus: 'installing',
      installStartedAt: new Date().toISOString(),
      installedComponents: []
    });
    return await install({
      esClient,
      soClient,
      logger,
      definition: entityDefinition
    });
  } catch (e) {
    logger.error(`Failed to install entity definition [${definition.id}]: ${e}`);
    await (0, _stop_transforms.stopLatestTransform)(esClient, definition, logger);
    await (0, _delete_transforms.deleteLatestTransform)(esClient, definition, logger);
    await (0, _delete_ingest_pipeline.deleteLatestIngestPipeline)(esClient, definition, logger);
    await (0, _manage_index_templates.deleteTemplate)({
      esClient,
      logger,
      name: (0, _generate_component_id.generateLatestIndexTemplateId)(definition)
    });
    await (0, _delete_entity_definition.deleteEntityDefinition)(soClient, definition).catch(err => {
      if (err instanceof _entity_not_found.EntityDefinitionNotFound) {
        return;
      }
      throw err;
    });
    throw e;
  }
}
async function installBuiltInEntityDefinitions({
  esClient,
  soClient,
  logger,
  definitions
}) {
  if (definitions.length === 0) return [];
  logger.info(`Checking installation of ${definitions.length} built-in definitions`);
  const installPromises = definitions.map(async builtInDefinition => {
    const installedDefinition = await (0, _find_entity_definition.findEntityDefinitionById)({
      soClient,
      esClient,
      id: builtInDefinition.id,
      includeState: true
    });
    if (!installedDefinition) {
      // clean data from previous installation
      await (0, _delete_index.deleteIndices)(esClient, builtInDefinition, logger);
      return await installEntityDefinition({
        definition: builtInDefinition,
        esClient,
        soClient,
        logger
      });
    }

    // verify existing installation
    if (!shouldReinstallBuiltinDefinition(installedDefinition, builtInDefinition)) {
      return installedDefinition;
    }
    logger.info(`Detected failed or outdated installation of definition [${installedDefinition.id}] v${installedDefinition.version}, installing v${builtInDefinition.version}`);
    return await reinstallEntityDefinition({
      soClient,
      esClient,
      logger,
      definition: installedDefinition,
      definitionUpdate: builtInDefinition,
      deleteData: true
    });
  });
  return await Promise.all(installPromises);
}

// perform installation of an entity definition components.
// assume definition saved object is already persisted
async function install({
  esClient,
  soClient,
  definition,
  logger
}) {
  logger.info(`Installing definition [${definition.id}] v${definition.version}`);
  logger.debug(() => JSON.stringify(definition, null, 2));
  logger.debug(`Installing index templates for definition [${definition.id}]`);
  const templates = await (0, _manage_index_templates.createAndInstallTemplates)(esClient, definition, logger);
  logger.debug(`Installing ingest pipelines for definition [${definition.id}]`);
  const pipelines = await (0, _create_and_install_ingest_pipeline.createAndInstallIngestPipelines)(esClient, definition, logger);
  logger.debug(`Installing transforms for definition [${definition.id}]`);
  const transforms = await (0, _create_and_install_transform.createAndInstallTransforms)(esClient, definition, logger);
  const updatedProps = await (0, _save_entity_definition.updateEntityDefinition)(soClient, definition.id, {
    installStatus: 'installed',
    installedComponents: [...templates, ...pipelines, ...transforms]
  });
  return {
    ...definition,
    ...updatedProps.attributes
  };
}

// stop and delete the current transforms and reinstall all the components
async function reinstallEntityDefinition({
  esClient,
  soClient,
  definition,
  definitionUpdate,
  logger,
  deleteData = false
}) {
  try {
    const updatedDefinition = (0, _merge_definition_update.mergeEntityDefinitionUpdate)(definition, definitionUpdate);
    logger.debug(() => `Reinstalling definition [${definition.id}] from v${definition.version} to v${definitionUpdate.version}\n${JSON.stringify(updatedDefinition, null, 2)}`);
    await (0, _save_entity_definition.updateEntityDefinition)(soClient, definition.id, {
      ...updatedDefinition,
      installStatus: 'upgrading',
      installStartedAt: new Date().toISOString()
    });
    logger.debug(`Deleting transforms for definition [${definition.id}] v${definition.version}`);
    await stopAndDeleteTransforms(esClient, definition, logger);
    if (deleteData) {
      await (0, _delete_index.deleteIndices)(esClient, definition, logger);
    }
    return await install({
      soClient,
      logger,
      esClient,
      definition: updatedDefinition
    });
  } catch (err) {
    await (0, _save_entity_definition.updateEntityDefinition)(soClient, definition.id, {
      installStatus: 'failed'
    });
    throw err;
  }
}
const INSTALLATION_TIMEOUT = 5 * 60 * 1000;
const installationInProgress = definition => {
  const {
    installStatus,
    installStartedAt
  } = definition;
  return (installStatus === 'installing' || installStatus === 'upgrading') && Date.now() - Date.parse(installStartedAt) < INSTALLATION_TIMEOUT;
};
exports.installationInProgress = installationInProgress;
const installationTimedOut = definition => {
  const {
    installStatus,
    installStartedAt
  } = definition;
  return (installStatus === 'installing' || installStatus === 'upgrading') && Date.now() - Date.parse(installStartedAt) >= INSTALLATION_TIMEOUT;
};
const shouldReinstallBuiltinDefinition = (installedDefinition, latestDefinition) => {
  const {
    installStatus,
    version,
    state
  } = installedDefinition;
  const timedOut = installationTimedOut(installedDefinition);
  const outdated = installStatus === 'installed' && _semver.default.neq(version, latestDefinition.version);
  const failed = installStatus === 'failed';
  const partial = installStatus === 'installed' && !state.installed;
  return timedOut || outdated || failed || partial;
};
const stopAndDeleteTransforms = async (esClient, definition, logger) => {
  await (0, _stop_transforms.stopTransforms)(esClient, definition, logger);
  await (0, _delete_transforms.deleteTransforms)(esClient, definition, logger);
};