"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CrawlerSingleDomainLogic = void 0;
var _kea = require("kea");
var _flash_messages = require("../../../shared/flash_messages");
var _http = require("../../../shared/http");
var _kibana = require("../../../shared/kibana");
var _routes = require("../../routes");
var _engine = require("../engine");
var _crawler_logic = require("./crawler_logic");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CrawlerSingleDomainLogic = exports.CrawlerSingleDomainLogic = (0, _kea.kea)({
  path: ['enterprise_search', 'app_search', 'crawler', 'crawler_single_domain'],
  actions: {
    deleteDomain: domain => ({
      domain
    }),
    fetchDomainData: domainId => ({
      domainId
    }),
    onReceiveDomainData: domain => ({
      domain
    }),
    updateCrawlRules: crawlRules => ({
      crawlRules
    }),
    updateEntryPoints: entryPoints => ({
      entryPoints
    }),
    updateSitemaps: sitemaps => ({
      sitemaps
    }),
    submitDeduplicationUpdate: (domain, {
      fields,
      enabled
    }) => ({
      domain,
      fields,
      enabled
    })
  },
  reducers: {
    dataLoading: [true, {
      onReceiveDomainData: () => false
    }],
    domain: [null, {
      onReceiveDomainData: (_, {
        domain
      }) => domain,
      updateCrawlRules: (currentDomain, {
        crawlRules
      }) => ({
        ...currentDomain,
        crawlRules
      }),
      updateEntryPoints: (currentDomain, {
        entryPoints
      }) => ({
        ...currentDomain,
        entryPoints
      }),
      updateSitemaps: (currentDomain, {
        sitemaps
      }) => ({
        ...currentDomain,
        sitemaps
      })
    }]
  },
  listeners: ({
    actions
  }) => ({
    deleteDomain: async ({
      domain
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        await http.delete(`/internal/app_search/engines/${engineName}/crawler/domains/${domain.id}`);
        _crawler_logic.CrawlerLogic.actions.fetchCrawlerData();
        (0, _flash_messages.flashSuccessToast)((0, _utils.getDeleteDomainSuccessMessage)(domain.url));
        _kibana.KibanaLogic.values.navigateToUrl((0, _engine.generateEnginePath)(_routes.ENGINE_CRAWLER_PATH));
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    fetchDomainData: async ({
      domainId
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      try {
        const response = await http.get(`/internal/app_search/engines/${engineName}/crawler/domains/${domainId}`);
        const domainData = (0, _utils.crawlerDomainServerToClient)(response);
        actions.onReceiveDomainData(domainData);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    },
    submitDeduplicationUpdate: async ({
      domain,
      fields,
      enabled
    }) => {
      const {
        http
      } = _http.HttpLogic.values;
      const {
        engineName
      } = _engine.EngineLogic.values;
      const payload = {
        deduplication_enabled: enabled,
        deduplication_fields: fields
      };
      try {
        const response = await http.put(`/internal/app_search/engines/${engineName}/crawler/domains/${domain.id}`, {
          body: JSON.stringify(payload)
        });
        const domainData = (0, _utils.crawlerDomainServerToClient)(response);
        actions.onReceiveDomainData(domainData);
      } catch (e) {
        (0, _flash_messages.flashAPIErrors)(e);
      }
    }
  })
});