"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteAttackDiscoverySchedulesRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _zodHelpers = require("@kbn/zod-helpers");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _build_response = require("../../../lib/build_response");
var _helpers = require("../../helpers");
var _is_feature_available = require("./utils/is_feature_available");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const deleteAttackDiscoverySchedulesRoute = router => {
  router.versioned.delete({
    access: 'internal',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_SCHEDULES_BY_ID,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        params: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.DeleteAttackDiscoverySchedulesRequestParams)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.DeleteAttackDiscoverySchedulesResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Check if scheduling feature available
    if (!(await (0, _is_feature_available.isFeatureAvailable)(ctx))) {
      return response.notFound();
    }

    // Perform license and authenticated user
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }
    const {
      id
    } = request.params;
    try {
      const dataClient = await assistantContext.getAttackDiscoverySchedulingDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      await dataClient.deleteSchedule({
        id
      });
      return response.ok({
        body: {
          id
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: {
          success: false,
          error: error.message
        },
        statusCode: error.statusCode
      });
    }
  });
};
exports.deleteAttackDiscoverySchedulesRoute = deleteAttackDiscoverySchedulesRoute;