"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.callAssistantGraph = void 0;
var _server = require("@kbn/langchain/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _telemetry = require("@kbn/langchain/server/tracers/telemetry");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _securityAiPrompts = require("@kbn/security-ai-prompts");
var _lodash = require("lodash");
var _generate_chat_title = require("./nodes/generate_chat_title");
var _tool_prompts = require("../../../prompt/tool_prompts");
var _local_prompt_object = require("../../../prompt/local_prompt_object");
var _helpers = require("../../../prompt/helpers");
var _utils = require("../../../../routes/utils");
var _prompt = require("../../../prompt");
var _prompts = require("./prompts");
var _graph = require("./graph");
var _helpers2 = require("./helpers");
var _helpers3 = require("../../../../ai_assistant_data_clients/anonymization_fields/helpers");
var _constants = require("../../../../../common/constants");
var _agentRunnable = require("./agentRunnable");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const callAssistantGraph = async ({
  abortSignal,
  assistantContext,
  actionsClient,
  alertsIndexPattern,
  assistantTools = [],
  connectorId,
  contentReferencesStore,
  conversationId,
  core,
  dataClients,
  esClient,
  inference,
  inferenceChatModelDisabled = false,
  langChainMessages,
  llmTasks,
  llmType,
  isOssModel,
  logger: parentLogger,
  isStream = false,
  onLlmResponse,
  onNewReplacements,
  replacements,
  request,
  savedObjectsClient,
  screenContext,
  size,
  systemPrompt,
  telemetry,
  telemetryParams,
  traceOptions,
  responseLanguage = 'English',
  timeout
}) => {
  var _dataClients$anonymiz, _await$dataClients$kb, _dataClients$kbDataCl, _traceOptions$project, _latestMessage$;
  const logger = parentLogger.get('defaultAssistantGraph');
  const isOpenAI = llmType === 'openai' && !isOssModel;
  const llmClass = (0, _utils.getLlmClass)(llmType);

  /**
   * Creates a new instance of llmClass.
   *
   * This function ensures that a new llmClass instance is created every time it is called.
   * This is necessary to avoid any potential side effects from shared state. By always
   * creating a new instance, we prevent other uses of llm from binding and changing
   * the state unintentionally. For this reason, only call createLlmInstance at runtime
   */
  const createLlmInstance = async () => !inferenceChatModelDisabled ? inference.getChatModel({
    request,
    connectorId,
    chatModelOptions: {
      model: request.body.model,
      signal: abortSignal,
      temperature: (0, _server.getDefaultArguments)(llmType).temperature,
      // prevents the agent from retrying on failure
      // failure could be due to bad connector, we should deliver that result to the client asap
      maxRetries: 0,
      telemetryMetadata: {
        pluginId: 'security_ai_assistant'
      }
      // TODO add timeout to inference once resolved https://github.com/elastic/kibana/issues/221318
      // timeout,
    }
  }) : new llmClass({
    actionsClient,
    connectorId,
    llmType,
    logger,
    // possible client model override,
    // let this be undefined otherwise so the connector handles the model
    model: request.body.model,
    // ensure this is defined because we default to it in the language_models
    // This is where the LangSmith logs (Metadata > Invocation Params) are set
    temperature: (0, _server.getDefaultArguments)(llmType).temperature,
    signal: abortSignal,
    streaming: isStream,
    // prevents the agent from retrying on failure
    // failure could be due to bad connector, we should deliver that result to the client asap
    maxRetries: 0,
    convertSystemMessageToHumanContent: false,
    timeout,
    telemetryMetadata: {
      pluginId: 'security_ai_assistant'
    }
  });
  const anonymizationFieldsRes = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$anonymiz = dataClients.anonymizationFieldsDataClient) === null || _dataClients$anonymiz === void 0 ? void 0 : _dataClients$anonymiz.findDocuments({
    perPage: 1000,
    page: 1
  }));
  const anonymizationFields = anonymizationFieldsRes ? (0, _helpers3.transformESSearchToAnonymizationFields)(anonymizationFieldsRes.data) : undefined;
  const latestMessage = langChainMessages.slice(-1); // the last message

  // Check if KB is available (not feature flag related)
  const isEnabledKnowledgeBase = (_await$dataClients$kb = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$kbDataCl = dataClients.kbDataClient) === null || _dataClients$kbDataCl === void 0 ? void 0 : _dataClients$kbDataCl.isInferenceEndpointExists())) !== null && _await$dataClients$kb !== void 0 ? _await$dataClients$kb : false;

  // Fetch any applicable tools that the source plugin may have registered
  const assistantToolParams = {
    alertsIndexPattern,
    assistantContext,
    anonymizationFields,
    connectorId,
    contentReferencesStore,
    esClient,
    inference,
    isEnabledKnowledgeBase,
    kbDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.kbDataClient,
    llmTasks,
    logger,
    onNewReplacements,
    replacements,
    request,
    size,
    telemetry,
    createLlmInstance,
    isOssModel
  };
  const tools = (await Promise.all(assistantTools.map(async tool => {
    let description;
    try {
      description = await (0, _securityAiPrompts.getPrompt)({
        actionsClient,
        connectorId,
        localPrompts: _tool_prompts.localToolPrompts,
        model: (0, _helpers.getModelOrOss)(llmType, isOssModel, request.body.model),
        promptId: tool.name,
        promptGroupId: _tool_prompts.promptGroupId,
        provider: llmType,
        savedObjectsClient
      });
    } catch (e) {
      logger.error(`Failed to get prompt for tool: ${tool.name}`);
    }
    const llm = await createLlmInstance();
    return tool.getTool({
      ...assistantToolParams,
      llm,
      isOssModel,
      description
    });
  }))).filter(e => e != null);
  const apmTracer = new _apm.APMTracer({
    projectName: (_traceOptions$project = traceOptions === null || traceOptions === void 0 ? void 0 : traceOptions.projectName) !== null && _traceOptions$project !== void 0 ? _traceOptions$project : 'default'
  }, logger);
  const telemetryTracer = telemetryParams ? new _telemetry.TelemetryTracer({
    // this line MUST come before kbTools are added
    elasticTools: tools.map(({
      name
    }) => name),
    telemetry,
    telemetryParams
  }, logger) : undefined;

  // If KB enabled, fetch for any KB IndexEntries and generate a tool for each
  if (isEnabledKnowledgeBase) {
    var _dataClients$kbDataCl2;
    const kbTools = await (dataClients === null || dataClients === void 0 ? void 0 : (_dataClients$kbDataCl2 = dataClients.kbDataClient) === null || _dataClients$kbDataCl2 === void 0 ? void 0 : _dataClients$kbDataCl2.getAssistantTools({
      esClient,
      contentReferencesStore
    }));
    if (kbTools) {
      tools.push(...kbTools);
    }
  }
  const defaultSystemPrompt = await (0, _prompt.getPrompt)({
    actionsClient,
    connectorId,
    model: (0, _helpers.getModelOrOss)(llmType, isOssModel, request.body.model),
    promptId: _prompt.promptDictionary.systemPrompt,
    promptGroupId: _local_prompt_object.promptGroupId.aiAssistant,
    provider: llmType,
    savedObjectsClient
  });
  const chatPromptTemplate = (0, _prompts.formatPrompt)({
    prompt: defaultSystemPrompt,
    additionalPrompt: systemPrompt,
    llmType
  });
  const llm = await createLlmInstance();
  const agentRunnable = await (0, _agentRunnable.agentRunnableFactory)({
    llm,
    llmType,
    tools,
    inferenceChatModelDisabled,
    isOpenAI,
    isStream,
    prompt: chatPromptTemplate
  });
  const {
    provider
  } = !llmType || llmType === 'inference' ? await (0, _securityAiPrompts.resolveProviderAndModel)({
    connectorId,
    actionsClient
  }) : {
    provider: llmType
  };
  const uiSettingsDateFormatTimezone = await core.uiSettings.client.get(_constants.DEFAULT_DATE_FORMAT_TZ);
  const assistantGraph = (0, _graph.getDefaultAssistantGraph)({
    agentRunnable,
    dataClients,
    // we need to pass it like this or streaming does not work for bedrock
    createLlmInstance,
    logger,
    actionsClient,
    savedObjectsClient,
    tools,
    replacements,
    // some chat models (bedrock) require a signal to be passed on agent invoke rather than the signal passed to the chat model
    ...(llmType === 'bedrock' ? {
      signal: abortSignal
    } : {}),
    getFormattedTime: () => (0, _helpers.getFormattedTime)({
      screenContextTimezone: screenContext === null || screenContext === void 0 ? void 0 : screenContext.timeZone,
      uiSettingsDateFormatTimezone
    }),
    telemetry,
    telemetryParams,
    contentReferencesStore
  });
  const inputs = {
    responseLanguage,
    conversationId,
    connectorId,
    llmType,
    isStream,
    isOssModel,
    input: (_latestMessage$ = latestMessage[0]) === null || _latestMessage$ === void 0 ? void 0 : _latestMessage$.content,
    provider: provider !== null && provider !== void 0 ? provider : ''
  };

  // make a fire and forget async call to generateChatTitle
  void (async () => {
    const model = await createLlmInstance();
    await (0, _generate_chat_title.generateChatTitle)({
      actionsClient,
      contentReferencesStore,
      conversationsDataClient: dataClients === null || dataClients === void 0 ? void 0 : dataClients.conversationsDataClient,
      logger,
      savedObjectsClient,
      state: {
        ...inputs
      },
      model,
      telemetryParams,
      telemetry
    }).catch(error => {
      logger.error(`Failed to generate chat title: ${error.message}`);
    });
  })();
  if (isStream) {
    var _telemetryParams$isEn;
    return (0, _helpers2.streamGraph)({
      apmTracer,
      assistantGraph,
      inputs,
      inferenceChatModelDisabled,
      isEnabledKnowledgeBase: (_telemetryParams$isEn = telemetryParams === null || telemetryParams === void 0 ? void 0 : telemetryParams.isEnabledKnowledgeBase) !== null && _telemetryParams$isEn !== void 0 ? _telemetryParams$isEn : false,
      logger,
      onLlmResponse,
      request,
      telemetry,
      telemetryTracer,
      traceOptions
    });
  }
  const graphResponse = await (0, _helpers2.invokeGraph)({
    apmTracer,
    assistantGraph,
    inputs,
    onLlmResponse,
    telemetryTracer,
    traceOptions
  });
  const {
    prunedContentReferencesStore,
    prunedContent
  } = (0, _elasticAssistantCommon.pruneContentReferences)(graphResponse.output, contentReferencesStore);
  const metadata = {
    ...(!(0, _lodash.isEmpty)(prunedContentReferencesStore) ? {
      contentReferences: prunedContentReferencesStore
    } : {})
  };
  return {
    body: {
      connector_id: connectorId,
      data: prunedContent,
      trace_data: graphResponse.traceData,
      replacements,
      status: 'ok',
      ...(!(0, _lodash.isEmpty)(metadata) ? {
        metadata
      } : {}),
      ...(graphResponse.conversationId ? {
        conversationId: graphResponse.conversationId
      } : {})
    },
    headers: {
      'content-type': 'application/json'
    }
  };
};
exports.callAssistantGraph = callAssistantGraph;