"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeFetch = initializeFetch;
exports.isEsqlMode = void 0;
var _rxjs = require("rxjs");
var _discoverUtils = require("@kbn/discover-utils");
var _esQuery = require("@kbn/es-query");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/inspector-plugin/common");
var _presentationPublishing = require("@kbn/presentation-publishing");
var _unifiedDataTable = require("@kbn/unified-data-table");
var _fetch_esql = require("../application/main/data_fetching/fetch_esql");
var _get_allowed_sample_size = require("../utils/get_allowed_sample_size");
var _initialize_edit_api = require("./initialize_edit_api");
var _update_search_source = require("./utils/update_search_source");
var _data_sources = require("../../common/data_sources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const isEsqlMode = savedSearch => {
  const query = savedSearch.searchSource.getField('query');
  return (0, _esQuery.isOfAggregateQueryType)(query);
};
exports.isEsqlMode = isEsqlMode;
const getExecutionContext = async (api, discoverServices) => {
  var _api$title$, _api$defaultTitle$;
  const {
    editUrl,
    urlWithoutLocationState
  } = await (0, _initialize_edit_api.getAppTarget)(api, discoverServices);
  const childContext = {
    type: _discoverUtils.SEARCH_EMBEDDABLE_TYPE,
    name: 'discover',
    id: api.savedObjectId$.getValue(),
    description: ((_api$title$ = api.title$) === null || _api$title$ === void 0 ? void 0 : _api$title$.getValue()) || ((_api$defaultTitle$ = api.defaultTitle$) === null || _api$defaultTitle$ === void 0 ? void 0 : _api$defaultTitle$.getValue()) || '',
    url: editUrl
  };
  const generateExecutionContext = createExecutionContext(api);
  const executionContext = generateExecutionContext(childContext);
  if (isExecutionContextWithinLimits(executionContext)) {
    return executionContext;
  }
  const newChildContext = {
    ...childContext,
    url: urlWithoutLocationState
  };
  return generateExecutionContext(newChildContext);
};
const createExecutionContext = api => childContext => {
  var _api$parentApi;
  return (0, _presentationPublishing.apiHasParentApi)(api) && (0, _presentationPublishing.apiHasExecutionContext)(api.parentApi) ? {
    ...((_api$parentApi = api.parentApi) === null || _api$parentApi === void 0 ? void 0 : _api$parentApi.executionContext),
    child: childContext
  } : childContext;
};
const isExecutionContextWithinLimits = executionContext => {
  const value = JSON.stringify(executionContext);
  const encoded = encodeURIComponent(value);

  // The max value is set to this arbitrary number because of the following reasons:
  // 1. Maximum allowed length of the `baggage` header via which the execution context is passed is 4096 / 4 = 1024 characters.
  // 2. The Execution Context Service adds labels (name, page and id) to the context additionally, which can increase the length
  // Hence as a safe limit, we set the maximum length of the execution context to 900 characters.
  const MAX_VALUE_ALLOWED = 900;
  return encoded.length < MAX_VALUE_ALLOWED;
};
function initializeFetch({
  api,
  stateManager,
  discoverServices,
  setDataLoading,
  setBlockingError
}) {
  const inspectorAdapters = {
    requests: new _common.RequestAdapter()
  };
  let abortController;
  const fetchSubscription = (0, _rxjs.combineLatest)([(0, _presentationPublishing.fetch$)(api), api.savedSearch$, api.dataViews$]).pipe((0, _rxjs.tap)(() => {
    // abort any in-progress requests
    if (abortController) {
      abortController.abort();
      abortController = undefined;
    }
  }), (0, _rxjs.switchMap)(async ([fetchContext, savedSearch, dataViews]) => {
    const dataView = dataViews !== null && dataViews !== void 0 && dataViews.length ? dataViews[0] : undefined;
    setBlockingError(undefined);
    if (!dataView || !savedSearch.searchSource) {
      return;
    }
    const useNewFieldsApi = !discoverServices.uiSettings.get(_discoverUtils.SEARCH_FIELDS_FROM_SOURCE, false);
    (0, _update_search_source.updateSearchSource)(discoverServices, savedSearch.searchSource, dataView, savedSearch.sort, (0, _get_allowed_sample_size.getAllowedSampleSize)(savedSearch.sampleSize, discoverServices.uiSettings), useNewFieldsApi, fetchContext, {
      sortDir: discoverServices.uiSettings.get(_discoverUtils.SORT_DEFAULT_ORDER_SETTING)
    });
    const searchSessionId = fetchContext.searchSessionId;
    const searchSourceQuery = savedSearch.searchSource.getField('query');

    // Log request to inspector
    inspectorAdapters.requests.reset();
    try {
      setDataLoading(true);

      // Get new abort controller
      const currentAbortController = new AbortController();
      abortController = currentAbortController;
      await discoverServices.profilesManager.resolveDataSourceProfile({
        dataSource: (0, _data_sources.createDataSource)({
          dataView,
          query: searchSourceQuery
        }),
        dataView,
        query: searchSourceQuery
      });
      const esqlMode = isEsqlMode(savedSearch);
      if (esqlMode && searchSourceQuery && (!fetchContext.query || (0, _esQuery.isOfQueryType)(fetchContext.query))) {
        // Request ES|QL data
        const result = await (0, _fetch_esql.fetchEsql)({
          query: searchSourceQuery,
          timeRange: (0, _update_search_source.getTimeRangeFromFetchContext)(fetchContext),
          inputQuery: fetchContext.query,
          filters: fetchContext.filters,
          dataView,
          abortSignal: currentAbortController.signal,
          inspectorAdapters,
          data: discoverServices.data,
          expressions: discoverServices.expressions,
          profilesManager: discoverServices.profilesManager
        });
        return {
          columnsMeta: result.esqlQueryColumns ? (0, _unifiedDataTable.getTextBasedColumnsMeta)(result.esqlQueryColumns) : undefined,
          rows: result.records,
          hitCount: result.records.length,
          fetchContext
        };
      }
      const executionContext = await getExecutionContext(api, discoverServices);

      /**
       * Fetch via saved search
       */
      const {
        rawResponse: resp
      } = await (0, _rxjs.lastValueFrom)(savedSearch.searchSource.fetch$({
        abortSignal: currentAbortController.signal,
        sessionId: searchSessionId,
        inspector: {
          adapter: inspectorAdapters.requests,
          title: _i18n.i18n.translate('discover.embeddable.inspectorTableRequestTitle', {
            defaultMessage: 'Table'
          }),
          description: _i18n.i18n.translate('discover.embeddable.inspectorRequestDescription', {
            defaultMessage: 'This request queries Elasticsearch to fetch the data for the search.'
          })
        },
        executionContext,
        disableWarningToasts: true
      }));
      const interceptedWarnings = [];
      discoverServices.data.search.showWarnings(inspectorAdapters.requests, warning => {
        interceptedWarnings.push(warning);
        return true; // suppress the default behaviour
      });
      return {
        warnings: interceptedWarnings,
        rows: (0, _discoverUtils.buildDataTableRecordList)({
          records: resp.hits.hits,
          dataView,
          processRecord: record => discoverServices.profilesManager.resolveDocumentProfile({
            record
          })
        }),
        hitCount: resp.hits.total,
        fetchContext
      };
    } catch (error) {
      return {
        error
      };
    }
  })).subscribe(next => {
    var _next$rows, _next$warnings;
    setDataLoading(false);
    if (!next || Object.hasOwn(next, 'error')) {
      setBlockingError(next === null || next === void 0 ? void 0 : next.error);
      return;
    }
    stateManager.rows.next((_next$rows = next.rows) !== null && _next$rows !== void 0 ? _next$rows : []);
    stateManager.totalHitCount.next(next.hitCount);
    stateManager.inspectorAdapters.next(inspectorAdapters);
    api.fetchWarnings$.next((_next$warnings = next.warnings) !== null && _next$warnings !== void 0 ? _next$warnings : []);
    api.fetchContext$.next(next.fetchContext);
    if (Object.hasOwn(next, 'columnsMeta')) {
      stateManager.columnsMeta.next(next.columnsMeta);
    }
  });
  return () => {
    fetchSubscription.unsubscribe();
  };
}