"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataTelemetryService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _register_collector = require("./register_collector");
var _constants = require("./constants");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SKIP_COLLECTION = 'Skip Collection';
class DataTelemetryService {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManagerStart", void 0);
    (0, _defineProperty2.default)(this, "telemetryStart", void 0);
    (0, _defineProperty2.default)(this, "usageCollection", void 0);
    (0, _defineProperty2.default)(this, "isOptedIn", true);
    // Assume true until the first check
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "isStopped", false);
    (0, _defineProperty2.default)(this, "isInProgress", false);
    (0, _defineProperty2.default)(this, "run$", (0, _rxjs.defer)(() => (0, _rxjs.from)(this.shouldCollectTelemetry())).pipe((0, _rxjs.switchMap)(isOptedIn => {
      // If stopped, do not proceed
      if (this.isStopped) {
        return this.throwSkipCollection();
      }
      return (0, _rxjs.of)(isOptedIn);
    }), (0, _rxjs.tap)(isOptedIn => {
      if (!isOptedIn) {
        this.logTelemetryNotOptedIn();
        this.isInProgress = false;
      } else {
        this.isInProgress = true;
      }
    }), (0, _rxjs.switchMap)(isOptedIn => {
      // If not opted in, do not proceed
      if (!isOptedIn) {
        return this.throwSkipCollection();
      }
      return (0, _rxjs.of)(isOptedIn);
    }), (0, _rxjs.exhaustMap)(() => {
      return this.collectTelemetryData();
    }), (0, _rxjs.tap)(() => this.isInProgress = false)));
    this.logger = logger;
  }
  setup(taskManagerSetup, usageCollection) {
    this.usageCollection = usageCollection;
    if (usageCollection) {
      // Register Kibana task
      this.registerTask(taskManagerSetup);
    } else {
      this.logger.warn(`[Logs Data Telemetry] Usage collection service is not available: cannot collect telemetry data`);
    }
  }
  async start(telemetryStart, core, taskManagerStart) {
    this.taskManagerStart = taskManagerStart;
    this.telemetryStart = telemetryStart;
    this.esClient = core === null || core === void 0 ? void 0 : core.elasticsearch.client.asInternalUser;
    if (taskManagerStart && this.usageCollection) {
      const taskInstance = await this.scheduleTask(taskManagerStart);
      if (taskInstance) {
        this.logger.debug(`Task ${taskInstance.id} scheduled.`);
      }

      // Create and register usage collector for logs data telemetry
      (0, _register_collector.registerLogsDataUsageCollector)(this.usageCollection, this.getCollectorOptions());
    }
  }
  stop() {
    this.isStopped = true;
  }
  resume() {
    this.isStopped = false;
  }
  registerTask(taskManager) {
    const service = this;
    taskManager.registerTaskDefinitions({
      [_constants.LOGS_DATA_TELEMETRY_TASK_TYPE]: {
        title: 'Logs Data Telemetry',
        description: 'This task collects data telemetry for logs data and sends it to the telemetry service via usage collector plugin.',
        timeout: `${_constants.TELEMETRY_TASK_TIMEOUT}m`,
        maxAttempts: 1,
        // Do not retry

        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            // Perform the work of the task. The return value should fit the TaskResult interface.
            async run() {
              var _state$data;
              const {
                state
              } = taskInstance;
              let data = (_state$data = state === null || state === void 0 ? void 0 : state.data) !== null && _state$data !== void 0 ? _state$data : null;
              try {
                data = await (0, _rxjs.firstValueFrom)(service.run$);
              } catch (e) {
                if (e.message === SKIP_COLLECTION) {
                  data = null; // Collection is skipped, skip reporting
                } else {
                  service.logger.error(e);
                }
              }
              return {
                state: {
                  ran: true,
                  data
                }
              };
            },
            async cancel() {
              service.logger.warn(`[Logs Data Telemetry] Task cancelled`);
            }
          };
        }
      }
    });
  }
  async scheduleTask(taskManager) {
    try {
      var _this$logger, _taskInstance$schedul;
      const taskInstance = await taskManager.ensureScheduled({
        id: _constants.LOGS_DATA_TELEMETRY_TASK_ID,
        taskType: _constants.LOGS_DATA_TELEMETRY_TASK_TYPE,
        schedule: {
          interval: `${_constants.TELEMETRY_TASK_INTERVAL}m`
        },
        params: {},
        state: {},
        scope: ['logs']
      });
      (_this$logger = this.logger) === null || _this$logger === void 0 ? void 0 : _this$logger.debug(`Task ${_constants.LOGS_DATA_TELEMETRY_TASK_ID} scheduled with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}.`);
      return taskInstance;
    } catch (e) {
      var _this$logger2;
      (_this$logger2 = this.logger) === null || _this$logger2 === void 0 ? void 0 : _this$logger2.error(`Failed to schedule task ${_constants.LOGS_DATA_TELEMETRY_TASK_ID} with interval ${_constants.TELEMETRY_TASK_INTERVAL}. ${e === null || e === void 0 ? void 0 : e.message}`);
      return null;
    }
  }
  async shouldCollectTelemetry() {
    var _this$telemetryStart;
    if (process.env.CI) {
      return false; // Telemetry collection flow should not run in CI
    }
    this.isOptedIn = await ((_this$telemetryStart = this.telemetryStart) === null || _this$telemetryStart === void 0 ? void 0 : _this$telemetryStart.getIsOptedIn());
    return this.isOptedIn === true;
  }
  collectTelemetryData() {
    // Gather data streams and indices related to each stream of log
    if (this.esClient) {
      return (0, _helpers.getAllIndices)({
        esClient: this.esClient,
        logsIndexPatterns: _constants.LOGS_DATASET_INDEX_PATTERNS,
        excludeStreamsStartingWith: [..._constants.NON_LOG_SIGNALS, ..._constants.EXCLUDE_ELASTIC_LOGS],
        breatheDelay: _constants.BREATHE_DELAY_MEDIUM
      }).pipe((0, _rxjs.switchMap)(dataStreamsAndIndicesInfo => {
        if (dataStreamsAndIndicesInfo.length > _constants.MAX_STREAMS_TO_REPORT) {
          this.logger.debug(`[Logs Data Telemetry] Number of data streams exceeds ${_constants.MAX_STREAMS_TO_REPORT}. Skipping telemetry collection.`);
          return this.throwSkipCollection();
        }
        return (0, _rxjs.of)(dataStreamsAndIndicesInfo);
      }), (0, _rxjs.delay)(_constants.BREATHE_DELAY_MEDIUM), (0, _rxjs.switchMap)(dataStreamsAndIndicesInfo => {
        return (0, _helpers.addMappingsToIndices)({
          esClient: this.esClient,
          dataStreamsInfo: dataStreamsAndIndicesInfo,
          logsIndexPatterns: _constants.LOGS_DATASET_INDEX_PATTERNS
        });
      }), (0, _rxjs.delay)(_constants.BREATHE_DELAY_SHORT), (0, _rxjs.switchMap)(dataStreamsAndIndicesInfo => {
        return (0, _helpers.addNamespace)({
          dataStreamsInfo: dataStreamsAndIndicesInfo
        });
      }), (0, _rxjs.delay)(_constants.BREATHE_DELAY_MEDIUM), (0, _rxjs.switchMap)(infoWithNamespace => {
        return (0, _helpers.getIndexBasicStats)({
          esClient: this.esClient,
          indices: infoWithNamespace,
          breatheDelay: _constants.BREATHE_DELAY_MEDIUM
        });
      }), (0, _rxjs.delay)(_constants.BREATHE_DELAY_SHORT), (0, _rxjs.switchMap)(infoWithStats => {
        return (0, _helpers.getIndexFieldStats)({
          basicStats: infoWithStats
        });
      }), (0, _rxjs.delay)(_constants.BREATHE_DELAY_SHORT), (0, _rxjs.map)(statsWithNamespace => {
        return (0, _helpers.groupStatsByPatternName)(statsWithNamespace);
      }), (0, _rxjs.map)(statsByPattern => {
        return (0, _helpers.indexStatsToTelemetryEvents)(statsByPattern);
      }));
    } else {
      this.logger.warn(`[Logs Data Telemetry] Elasticsearch client is unavailable: cannot retrieve data streams
        for stream of logs`);
      return this.throwSkipCollection();
    }
  }
  getCollectorOptions() {
    return {
      fetch: async () => {
        var _taskState$data;
        // Retrieve the latest telemetry data from task manager
        const taskState = await this.getLatestTaskState();
        return {
          data: (_taskState$data = taskState.data) !== null && _taskState$data !== void 0 ? _taskState$data : []
        };
      },
      isReady: async () => {
        const taskState = await this.getLatestTaskState();
        return !this.isStopped && !this.isInProgress && taskState.ran && taskState.data !== null;
      }
    };
  }
  async getLatestTaskState() {
    const defaultState = {
      data: null,
      ran: false
    };
    if (this.taskManagerStart) {
      try {
        var _fetchResult$docs$0$s, _fetchResult$docs$;
        const fetchResult = await this.taskManagerStart.fetch({
          query: {
            bool: {
              filter: {
                term: {
                  _id: `task:${_constants.LOGS_DATA_TELEMETRY_TASK_ID}`
                }
              }
            }
          }
        });
        return (_fetchResult$docs$0$s = (_fetchResult$docs$ = fetchResult.docs[0]) === null || _fetchResult$docs$ === void 0 ? void 0 : _fetchResult$docs$.state) !== null && _fetchResult$docs$0$s !== void 0 ? _fetchResult$docs$0$s : defaultState;
      } catch (err) {
        const errMessage = err && err.message ? err.message : err.toString();
        if (!errMessage.includes('NotInitialized')) {
          throw err;
        }
      }
    } else {
      this.logger.error(`[Logs Data Telemetry] Task manager is not available: cannot retrieve latest task state`);
    }
    return defaultState;
  }
  logTelemetryNotOptedIn() {
    this.logger.debug(`[Logs Data Telemetry] Telemetry is not opted-in.`);
  }
  throwSkipCollection() {
    return (0, _rxjs.throwError)(() => new Error(SKIP_COLLECTION));
  }
}
exports.DataTelemetryService = DataTelemetryService;