"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DashboardStorage = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _contentManagementUtils = require("@kbn/content-management-utils");
var _dashboard_saved_object = require("../dashboard_saved_object");
var _cm_services = require("./cm_services");
var _latest = require("./latest");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getRandomColor = () => {
  return '#' + String(Math.floor(Math.random() * 16777215).toString(16)).padStart(6, '0');
};
const searchArgsToSOFindOptions = (query, options) => {
  return {
    type: _dashboard_saved_object.DASHBOARD_SAVED_OBJECT_TYPE,
    searchFields: options !== null && options !== void 0 && options.onlyTitle ? ['title'] : ['title^3', 'description'],
    fields: options === null || options === void 0 ? void 0 : options.fields,
    search: query.text,
    perPage: query.limit,
    page: query.cursor ? +query.cursor : undefined,
    defaultSearchOperator: 'AND',
    namespaces: options === null || options === void 0 ? void 0 : options.spaces,
    ...(0, _contentManagementUtils.tagsToFindOptions)(query.tags)
  };
};
const savedObjectClientFromRequest = async ctx => {
  if (!ctx.requestHandlerContext) {
    throw new Error('Storage context.requestHandlerContext missing.');
  }
  const {
    savedObjects
  } = await ctx.requestHandlerContext.core;
  return savedObjects.client;
};
class DashboardStorage {
  constructor({
    logger,
    throwOnResultValidationError,
    savedObjectsTagging
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsTagging", void 0);
    (0, _defineProperty2.default)(this, "throwOnResultValidationError", void 0);
    this.savedObjectsTagging = savedObjectsTagging;
    this.logger = logger;
    this.throwOnResultValidationError = throwOnResultValidationError !== null && throwOnResultValidationError !== void 0 ? throwOnResultValidationError : false;
  }
  getTagNamesFromReferences(references, allTags) {
    return Array.from(new Set(this.savedObjectsTagging ? this.savedObjectsTagging.getTagsFromReferences(references, allTags).tags.map(tag => tag.name) : []));
  }
  getUniqueTagNames(references, newTagNames, allTags) {
    const referenceTagNames = this.getTagNamesFromReferences(references, allTags);
    return new Set([...referenceTagNames, ...newTagNames]);
  }
  async replaceTagReferencesByName(references, newTagNames, allTags, tagsClient) {
    var _this$savedObjectsTag, _this$savedObjectsTag2;
    const combinedTagNames = this.getUniqueTagNames(references, newTagNames, allTags);
    const newTagIds = await this.convertTagNamesToIds(combinedTagNames, allTags, tagsClient);
    return (_this$savedObjectsTag = (_this$savedObjectsTag2 = this.savedObjectsTagging) === null || _this$savedObjectsTag2 === void 0 ? void 0 : _this$savedObjectsTag2.replaceTagReferences(references, newTagIds)) !== null && _this$savedObjectsTag !== void 0 ? _this$savedObjectsTag : references;
  }
  async convertTagNamesToIds(tagNames, allTags, tagsClient) {
    const combinedTagNames = await this.createTagsIfNeeded(tagNames, allTags, tagsClient);
    return Array.from(combinedTagNames).flatMap(tagName => {
      var _this$savedObjectsTag3, _this$savedObjectsTag4;
      return (_this$savedObjectsTag3 = (_this$savedObjectsTag4 = this.savedObjectsTagging) === null || _this$savedObjectsTag4 === void 0 ? void 0 : _this$savedObjectsTag4.convertTagNameToId(tagName, allTags)) !== null && _this$savedObjectsTag3 !== void 0 ? _this$savedObjectsTag3 : [];
    });
  }
  async createTagsIfNeeded(tagNames, allTags, tagsClient) {
    const tagsToCreate = Array.from(tagNames).filter(tagName => !allTags.some(tag => tag.name === tagName));
    const tagCreationResults = await Promise.allSettled(tagsToCreate.flatMap(tagName => {
      var _tagsClient$create;
      return (_tagsClient$create = tagsClient === null || tagsClient === void 0 ? void 0 : tagsClient.create({
        name: tagName,
        description: '',
        color: getRandomColor()
      })) !== null && _tagsClient$create !== void 0 ? _tagsClient$create : [];
    }));
    for (const result of tagCreationResults) {
      if (result.status === 'rejected') {
        this.logger.error(`Error creating tag: ${result.reason}`);
      } else {
        this.logger.info(`Tag created: ${result.value.name}`);
      }
    }
    const createdTags = tagCreationResults.filter(result => result.status === 'fulfilled').map(result => result.value);

    // Remove tags that were not created
    const invalidTagNames = tagsToCreate.filter(tagName => !createdTags.some(tag => tag.name === tagName));
    invalidTagNames.forEach(tagName => tagNames.delete(tagName));

    // Add newly created tags to allTags
    allTags.push(...createdTags);
    const combinedTagNames = new Set([...tagNames, ...createdTags.map(createdTag => createdTag.name)]);
    return combinedTagNames;
  }
  async get(ctx, id) {
    var _this$savedObjectsTag5, _await$tagsClient$get;
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const tagsClient = (_this$savedObjectsTag5 = this.savedObjectsTagging) === null || _this$savedObjectsTag5 === void 0 ? void 0 : _this$savedObjectsTag5.createTagClient({
      client: soClient
    });
    const allTags = (_await$tagsClient$get = await (tagsClient === null || tagsClient === void 0 ? void 0 : tagsClient.getAll())) !== null && _await$tagsClient$get !== void 0 ? _await$tagsClient$get : [];
    // Save data in DB
    const {
      saved_object: savedObject,
      alias_purpose: aliasPurpose,
      alias_target_id: aliasTargetId,
      outcome
    } = await soClient.resolve(_dashboard_saved_object.DASHBOARD_SAVED_OBJECT_TYPE, id);
    const {
      item,
      error: itemError
    } = (0, _latest.savedObjectToItem)(savedObject, false, {
      getTagNamesFromReferences: references => this.getTagNamesFromReferences(references, allTags)
    });
    if (itemError) {
      throw _boom.default.badRequest(`Invalid response. ${itemError.message}`);
    }
    const response = {
      item,
      meta: {
        aliasPurpose,
        aliasTargetId,
        outcome
      }
    };
    const validationError = transforms.get.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.get.out.result.down(response, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async bulkGet() {
    // Not implemented
    throw new Error(`[bulkGet] has not been implemented. See DashboardStorage class.`);
  }
  async create(ctx, data, options) {
    var _this$savedObjectsTag6;
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const tagsClient = (_this$savedObjectsTag6 = this.savedObjectsTagging) === null || _this$savedObjectsTag6 === void 0 ? void 0 : _this$savedObjectsTag6.createTagClient({
      client: soClient
    });
    const allTags = tagsClient ? await (tagsClient === null || tagsClient === void 0 ? void 0 : tagsClient.getAll()) : [];

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.create.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.create.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes: soAttributes,
      references: soReferences,
      error: attributesError
    } = await (0, _latest.itemAttrsToSavedObjectWithTags)({
      attributes: dataToLatest,
      replaceTagReferencesByName: ({
        references,
        newTagNames
      }) => this.replaceTagReferencesByName(references, newTagNames, allTags, tagsClient),
      incomingReferences: options.references
    });
    if (attributesError) {
      throw _boom.default.badRequest(`Invalid data. ${attributesError.message}`);
    }

    // Save data in DB
    const savedObject = await soClient.create(_dashboard_saved_object.DASHBOARD_SAVED_OBJECT_TYPE, soAttributes, {
      ...optionsToLatest,
      references: soReferences
    });
    const {
      item,
      error: itemError
    } = (0, _latest.savedObjectToItem)(savedObject, false, {
      getTagNamesFromReferences: references => this.getTagNamesFromReferences(references, allTags)
    });
    if (itemError) {
      throw _boom.default.badRequest(`Invalid response. ${itemError.message}`);
    }
    const validationError = transforms.create.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.create.out.result.down({
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async update(ctx, id, data, options) {
    var _this$savedObjectsTag7, _await$tagsClient$get2;
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const tagsClient = (_this$savedObjectsTag7 = this.savedObjectsTagging) === null || _this$savedObjectsTag7 === void 0 ? void 0 : _this$savedObjectsTag7.createTagClient({
      client: soClient
    });
    const allTags = (_await$tagsClient$get2 = await (tagsClient === null || tagsClient === void 0 ? void 0 : tagsClient.getAll())) !== null && _await$tagsClient$get2 !== void 0 ? _await$tagsClient$get2 : [];

    // Validate input (data & options) & UP transform them to the latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.update.in.data.up(data);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.update.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const {
      attributes: soAttributes,
      references: soReferences,
      error: attributesError
    } = await (0, _latest.itemAttrsToSavedObjectWithTags)({
      attributes: dataToLatest,
      replaceTagReferencesByName: ({
        references,
        newTagNames
      }) => this.replaceTagReferencesByName(references, newTagNames, allTags, tagsClient),
      incomingReferences: options.references
    });
    if (attributesError) {
      throw _boom.default.badRequest(`Invalid data. ${attributesError.message}`);
    }

    // Save data in DB
    const partialSavedObject = await soClient.update(_dashboard_saved_object.DASHBOARD_SAVED_OBJECT_TYPE, id, soAttributes, {
      ...optionsToLatest,
      references: soReferences
    });
    const {
      item,
      error: itemError
    } = (0, _latest.savedObjectToItem)(partialSavedObject, true, {
      getTagNamesFromReferences: references => this.getTagNamesFromReferences(references, allTags)
    });
    if (itemError) {
      throw _boom.default.badRequest(`Invalid response. ${itemError.message}`);
    }
    const validationError = transforms.update.out.result.validate({
      item
    });
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate DB response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.update.out.result.down({
      item
    }, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async delete(ctx, id,
  // force is necessary to delete saved objects that exist in multiple namespaces
  options) {
    var _options$force;
    const soClient = await savedObjectClientFromRequest(ctx);
    await soClient.delete(_dashboard_saved_object.DASHBOARD_SAVED_OBJECT_TYPE, id, {
      force: (_options$force = options === null || options === void 0 ? void 0 : options.force) !== null && _options$force !== void 0 ? _options$force : false
    });
    return {
      success: true
    };
  }
  async search(ctx, query, options) {
    var _this$savedObjectsTag8, _await$tagsClient$get3;
    const transforms = ctx.utils.getTransforms(_cm_services.cmServicesDefinition);
    const soClient = await savedObjectClientFromRequest(ctx);
    const tagsClient = (_this$savedObjectsTag8 = this.savedObjectsTagging) === null || _this$savedObjectsTag8 === void 0 ? void 0 : _this$savedObjectsTag8.createTagClient({
      client: soClient
    });
    const allTags = (_await$tagsClient$get3 = await (tagsClient === null || tagsClient === void 0 ? void 0 : tagsClient.getAll())) !== null && _await$tagsClient$get3 !== void 0 ? _await$tagsClient$get3 : [];

    // Validate and UP transform the options
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.search.in.options.up(options);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid payload. ${optionsError.message}`);
    }
    const soQuery = searchArgsToSOFindOptions(query, optionsToLatest);
    // Execute the query in the DB
    const soResponse = await soClient.find(soQuery);
    const hits = soResponse.saved_objects.map(so => {
      const {
        item
      } = (0, _latest.savedObjectToItem)(so, false, {
        allowedAttributes: soQuery.fields,
        allowedReferences: optionsToLatest === null || optionsToLatest === void 0 ? void 0 : optionsToLatest.includeReferences,
        getTagNamesFromReferences: references => this.getTagNamesFromReferences(references, allTags)
      });
      return item;
    })
    // Ignore any saved objects that failed to convert to items.
    .filter(item => item !== null);
    const response = {
      hits,
      pagination: {
        total: soResponse.total
      }
    };
    const validationError = transforms.search.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // Validate the response and DOWN transform to the request version
    const {
      value,
      error: resultError
    } = transforms.search.out.result.down(response, undefined,
    // do not override version
    {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
}
exports.DashboardStorage = DashboardStorage;