"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Coordinator = void 0;
exports.notificationCoordinator = notificationCoordinator;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class Coordinator {
  constructor() {
    (0, _defineProperty2.default)(this, "coordinationLock$", new _rxjs.BehaviorSubject({
      locked: false,
      controller: null
    }));
    (0, _defineProperty2.default)(this, "lock$", this.coordinationLock$.asObservable());
  }
  acquireLock(owner) {
    const {
      locked,
      controller
    } = this.coordinationLock$.getValue();
    if (locked && controller !== owner) {
      throw new Error(_i18n.i18n.translate('core.notifications.notificationLockError', {
        defaultMessage: 'Notification lock is already acquired'
      }));
    }
    this.coordinationLock$.next({
      locked: true,
      controller: owner
    });
  }
  releaseLock(owner) {
    const {
      locked,
      controller
    } = this.coordinationLock$.getValue();
    if (locked && controller === owner) {
      this.coordinationLock$.next({
        locked: false,
        controller: null
      });
    }
  }

  /**
   * @description This method is used to control the flow of notifications, it will automatically acquire
   * a lock for the registered entity and buffer new values to be emitted when the provided condition is not met.
   * Also when the passed observable has been emptied if the current lock still belongs to the registrar it will be released automatically, however an acquired lock can
   * also be released as deemed fit using {@link releaseLock}.
   * @param registrar - A unique identifier for the caller of this method
   * @param $ - Observable to be controlled
   * @param cond - Condition under which updates from the provided observable should be emitted
   */
  optInToCoordination(registrar, $, cond) {
    // signal used to determine when to emit values from the source observable based on the provided opt-in condition
    const on$ = this.coordinationLock$.pipe((0, _rxjs.filter)(state => cond(state)));
    // signal used to determine when to buffer values from the source observable based on the provided opt-in condition
    const off$ = this.coordinationLock$.pipe((0, _rxjs.filter)(state => !cond(state)));
    const multicast$ = $.pipe((0, _rxjs.share)());
    return (0, _rxjs.merge)(multicast$.pipe((0, _rxjs.bufferToggle)(off$, () => on$)), multicast$.pipe((0, _rxjs.windowToggle)(on$, () => off$))).pipe((0, _rxjs.mergeMap)(x => x), (0, _rxjs.tap)(value => {
      const lock = this.coordinationLock$.getValue();
      // if the source has values to emit and the lock is not owned (because of values that were buffered whilst the lock was owned by another registrar) we acquire a lock for said source.
      if (value.length && !lock.locked) {
        this.acquireLock(registrar);
      } else if (!value.length && lock.controller === registrar) {
        // here we handle the scenario where the source observable has been emptied out,
        // in such event if the lock is still held by the registrar we release it
        this.releaseLock(registrar);
      }
    }), (0, _rxjs.finalize)(() => {
      // when the coordinated observable is completed we release the lock if it is still held by the registrar
      const lock = this.coordinationLock$.getValue();
      if (lock.locked && lock.controller === registrar) {
        this.releaseLock(registrar);
      }
    }));
  }
}
exports.Coordinator = Coordinator;
function notificationCoordinator(registrar) {
  return {
    optInToCoordination: ($, cond) => this.optInToCoordination.apply(this, [registrar, $, cond]),
    acquireLock: this.acquireLock.bind(this, registrar),
    releaseLock: this.releaseLock.bind(this, registrar),
    lock$: this.lock$
  };
}